//LIC// FLXLab v2.5 - A program for running psychology experiments.  
//LIC// Copyright (C) 2010 Todd R. Haskell (todd.haskell@wwu.edu) 
//LIC// 
//LIC// Use and distribution is governed by the terms of the 
//LIC// GNU General Public License. Certain portions of the 
//LIC// program may be subject to other licenses as well. See 
//LIC// the file LICENSE.TXT for details.
//LIC// 
#ifndef FLXBASE_FLX_SETTING
#define FLXBASE_FLX_SETTING

#include "flxbase\FlxObject.h"
#include "flxbase\object_manager.h"
#include "flxbase\string_utils.h"
#include "flxbase\logical_and_numeric_types.h"
#include "flxbase\scopes.h"
#include "flxbase\linkage.h"

FLX_LINKAGE extern bool flx_setting_updates_flag;

FLX_LINKAGE extern void flx_suspend_setting_updates(void);
FLX_LINKAGE extern void flx_resume_setting_updates(void);

/*****************************************************************************/

template <class T>
class FlxSetting : public FlxObject {
 protected:
  T *d_setting_ptr;
  T d_old_val;
  string d_label;
 public:
 FlxSetting(T *setting_ptr,T value,string label) : FlxObject("setting:"+flx_unique_name()), d_setting_ptr(setting_ptr), d_old_val(*d_setting_ptr), d_label(label) {
    string cur_function="FlxSetting::FlxSetting";

    flx_data->write_message(FLX_DATADDEBUG,cur_function,"Changing setting of '"+label+"' from '"+flx_convert_to_string(d_old_val)+"' to '"+flx_convert_to_string(value)+"'");

    /* Change the setting */
    *d_setting_ptr=value;
  }
  virtual ~FlxSetting(void){
    string cur_function="FlxSetting::~FlxSetting";
    
    flx_data->write_message(FLX_DATADDEBUG,cur_function,"Reverting setting of '"+d_label+"' from '"+flx_convert_to_string(*d_setting_ptr)+"' to '"+flx_convert_to_string(d_old_val)+"'");

    /* Revert the setting itself */
    *d_setting_ptr=d_old_val;
  }
};

/*****************************************************************************/

template <class T>
class FlxClassSetting : public FlxSetting<T> {
  FlxObject *d_dependent_obj;
  FlxObject *d_old_source_obj;
  FlxObject *d_new_source_obj;
  static bool d_updates_flag;
 public:
 FlxClassSetting(FlxObject *dependent_obj,FlxObject *old_source_obj,FlxObject *new_source_obj,T *setting_ptr,T value,string label) : FlxSetting<T>(setting_ptr,value,label), d_dependent_obj(dependent_obj), d_old_source_obj(old_source_obj), d_new_source_obj(new_source_obj) {
    string cur_function="FlxObjectSetting::FlxObjectSetting";
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Updating dependencies for object '"+d_dependent_obj->name()+"', setting '"+this->d_label+"' prior to changing the setting itself.");

    /* Remove the old dependency, if there is one */
    if(d_old_source_obj){
      d_old_source_obj->remove_dependent(d_dependent_obj);
      d_dependent_obj->remove_source(d_old_source_obj);
    }

    /* Add the new dependency */
    d_new_source_obj->add_dependent(d_dependent_obj);
    d_dependent_obj->add_source(d_new_source_obj);

    /* Update the dependent, if appropriate */
    if(flx_setting_updates_flag){
      flx_process_dependencies(d_dependent_obj);
    }
  }
  ~FlxClassSetting(){
    string cur_function="FlxClassSetting::~FlxClassSetting";

    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Reverting dependencies for object '"+d_dependent_obj->name()+"', setting '"+this->d_label+"' prior to reverting the setting itself.");

    /* Remove the current dependency */
    d_new_source_obj->remove_dependent(d_dependent_obj);
    d_dependent_obj->remove_source(d_new_source_obj);
    
    /* Re-create the old dependency, if there was one */
    if(d_old_source_obj){
      d_old_source_obj->add_dependent(d_dependent_obj);
      d_dependent_obj->add_source(d_old_source_obj);
    }

    /* If the dependent has a different scope than the setting itself,
       such that the dependent will continue to exist even though
       the setting is gone, then we need to check whether the
       dependent should be updated to reflect the previous setting */
    if(d_dependent_obj->scope()!=this->scope()){
      /* Revert the setting; the destructor for FlxSetting will do
       this as well, but if we're going to update the object, we need
       to revert the setting before we update, or else the changes
       won't be reflected. There's no harm in doing the reverting
       twice, although it's not elegant or efficient. */
      *(this->d_setting_ptr)=this->d_old_val;
      if(flx_setting_updates_flag){
	flx_process_dependencies(d_dependent_obj);
      }
    }
  }
};

/*****************************************************************************/

template <class T>
void flx_change_global_setting(T *setting,T value,string label){
  string cur_function="flx_change_global_setting";
  
  new FlxSetting<T>(setting,value,label);

} /* flx_change_global_setting */

/*****************************************************************************/

template <class T>
void flx_change_object_setting(FlxObject *dependent_obj,T *&setting,T *value,string label){
  string cur_function="flx_change_object_setting";
  FlxObject *old_source_obj, *new_source_obj;

  flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Preparing to change object setting for '"+label+"'");

  if(setting==NULL){
    old_source_obj=NULL;
  } else if(!(old_source_obj=dynamic_cast<FlxObject *>(setting))){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Old setting value isn't a FlxObject");
    return;
  }

  if(!(new_source_obj=dynamic_cast<FlxObject *>(value))){
    flx_data->write_message(FLX_DATAERROR,cur_function,"New setting value isn't a FlxObject");
    return;
  }
  
  new FlxClassSetting<T *>(dependent_obj,old_source_obj,new_source_obj,&setting,value,label);

} /* flx_change_object_setting */

/*****************************************************************************/

template <class T>
void flx_change_scalar_setting(FlxObject *dependent_obj,T *&setting,T *value,string label){
  string cur_function="flx_change_scalar_setting";
  FlxObject *old_source_obj, *new_source_obj;

  flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Preparing to change scalar setting for '"+label+"'");

  if(setting==NULL){
    old_source_obj=NULL;
  } else if(!(old_source_obj=flx_get_object_by_value(setting))){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Old setting value isn't in the object list");
    return;
  }

  if(!(new_source_obj=flx_get_object_by_value(value))){
    flx_data->write_message(FLX_DATAERROR,cur_function,"New setting value isn't in the object list");
    return;
  }

  new FlxClassSetting<T *>(dependent_obj,old_source_obj,new_source_obj,&setting,value,label);
  
} /* flx_change_scalar_setting */

/*****************************************************************************/

#endif
