//LIC// FLXLab v2.5 - A program for running psychology experiments.  
//LIC// Copyright (C) 2010 Todd R. Haskell (todd.haskell@wwu.edu) 
//LIC// 
//LIC// Use and distribution is governed by the terms of the 
//LIC// GNU General Public License. Certain portions of the 
//LIC// program may be subject to other licenses as well. See 
//LIC// the file LICENSE.TXT for details.
//LIC// 
#ifndef FLXBASE_OBJECT_MANAGER
#define FLXBASE_OBJECT_MANAGER

#include <string>
#include <typeinfo>
#include "flxbase\FlxDataSystem.h"
#include "flxbase\FlxObject.h"

/* We need this so we don't have to qualify all STL names with
"std::" */
using namespace std;

FLX_LINKAGE extern string flx_unique_name(void);

template <class T> FlxObject *flx_get_object_by_value(T *val);

/*****************************************************************************/

class FLX_LINKAGE FlxBaseScalarObject : public FlxObject {
 public:
 FlxBaseScalarObject(const string &name) : FlxObject(name) {}
  ~FlxBaseScalarObject(void){}
  virtual FlxBaseScalarObject *create_from_string(const string &s){ return NULL; }
  virtual bool compare(FlxBaseScalarObject *sop){ return false; }
};

/*****************************************************************************/

template <class T>
class FlxScalarObject : public FlxBaseScalarObject {
 protected:
  T *d_value;
  bool d_object_frees_value;
 public:
 FlxScalarObject(const string &name) : FlxBaseScalarObject(name), d_value(NULL), d_object_frees_value(false) {}
  virtual ~FlxScalarObject(void){ if(d_object_frees_value) delete d_value; }
  void set_value(T *value){ 
    if(d_object_frees_value){
      delete d_value;
      d_object_frees_value=false;
    }
    d_value=value; 
  }
  /* with this version the object handles the memory management */
  void set_value(T value){ 
    if(d_object_frees_value){
      delete d_value;
    }
    d_value=new T(value); 
    d_object_frees_value=true;
  }
  T *value(void){ return d_value; }
  bool compare(FlxBaseScalarObject *bop){
    FlxScalarObject<T> *sop;

    if((sop=dynamic_cast<FlxScalarObject<T> *>(bop))){
      return *d_value==*(sop->d_value);
    } else {
      return false;
    }
  }
  FlxBaseScalarObject *create_from_string(const string &s){
    FlxScalarObject<T> *sop;
    T *vp=NULL;
    
    if(flx_type_convert(s,vp)){
      sop=dynamic_cast<FlxScalarObject<T> *>(flx_get_object_by_value<T>(vp));
      return sop;
    } else {
      return NULL;
    }
  }
  string value_as_string(void){ return flx_convert_to_string(*d_value); }
};

/*****************************************************************************/

template <class T>
class FlxScalarFromStringObject : public FlxScalarObject<T> {
 protected:
  string *d_source_string;
 public:
 FlxScalarFromStringObject(const string &name,string *source_string) : FlxScalarObject<T>(name), d_source_string(source_string) { this->d_value=new T(); }
  ~FlxScalarFromStringObject(void){ delete this->d_value; }
  bool update(void){ return flx_type_convert(*d_source_string,this->d_value); }
};

/*****************************************************************************/

template <class T>
T *flx_get_object_by_name(const string &name){
  string cur_function="flx_get_object_by_name";
  FlxObject **obj_handle;
  T *obj_ptr;

  if((obj_handle=flx_object_list.find_name(name))){
    // check if the object is of the right type
    if((obj_ptr=dynamic_cast<T *>(*obj_handle))){
      return obj_ptr;
    } else {
      // there's an object with the right name, but it has the wrong type
      return NULL;
    }
  } else {
    // no object by that name
    return NULL;
  }

} /* flx_get_object_by_name */

/*****************************************************************************/

template <class T>
FlxObject *flx_get_object_by_value(T *value){
  FlxObject **obj_handle;
  FlxScalarObject<T> *obj_ptr, *match_ptr=NULL;
  string cur_function="flx_get_object_by_value";

  flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Looking up object by value '"+flx_convert_to_string(value)+"'");
  while((obj_handle=flx_object_list.item())){
    if((obj_ptr=dynamic_cast<FlxScalarObject<T> *>(*obj_handle))){
      if(obj_ptr->value()==value){
	match_ptr=obj_ptr;
	break;
      }
    }
    flx_object_list.advance();
  }
  flx_object_list.rewind();
  return match_ptr;

} /* flx_get_object_by_value */

/*****************************************************************************/

template <class T>
T *flx_implicit_scalar_conversion(const string &name){
  string cur_function="flx_implicit_scalar_conversion";
  FlxScalarObject<string> *sop;
  FlxScalarFromStringObject<T> *obj_ptr;
  string new_name;

  sop=flx_get_object_by_name<FlxScalarObject<string> >(name);
  if(!sop){
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Implicit conversion failed, no string variable named '"+name+"'");
    return NULL;
  } else {
    // there is a string variable available
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Doing implicit conversion from a string object");
    // create the FlxScalarFromStringObject
    new_name=flx_unique_name();
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Creating object of type FlxScalarFromStringObject named '"+new_name+"'");
    obj_ptr=new FlxScalarFromStringObject<T>(new_name,sop->value());
    flx_add_object_source(obj_ptr,sop);
    return obj_ptr->value();
  }

} /* flx_implicit_scalar_conversion */

/*****************************************************************************/

template <class T>
T *flx_implicit_object_conversion(const string &name){
  string cur_function="flx_implicit_object_conversion";
  FlxScalarObject<string> *sop;
  string new_name;
  T *obj_ptr;

  sop=flx_get_object_by_name<FlxScalarObject<string> >(name);
  if(!sop){
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Implicit conversion failed, no string variable named '"+name+"'");
    return NULL;
  } else {
    // there is a string variable available
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Doing implicit conversion from a string object");
    // create the new implicit object
    new_name=flx_unique_name();
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Creating object named '"+new_name+"'");
    obj_ptr=new T(new_name,sop->value());
    flx_add_object_source(obj_ptr,sop);
    return obj_ptr;
  }

} /* flx_implicit_object_conversion */

/*****************************************************************************/

FLX_LINKAGE extern void flx_add_object_source(FlxObject *dependent,FlxObject *source);

template <class A>
void flx_add_scalar_source(FlxObject *dependent,A *source){
  string cur_function="flx_add_scalar_source";
  FlxObject *sop;
  sop=flx_get_object_by_value(source);
if(!sop){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Error adding a dependent by value, source can't be found in the object list");
  }
  flx_add_object_source(dependent,sop);
}

/*****************************************************************************/

template <class T>
T *flx_X(T val){
  FlxScalarObject<T> *vop;
  string name;
  string cur_function="flx_X<T>";
  T *value_ptr;

  name=typeid(T).name()+flx_unique_name();
  vop=new FlxScalarObject<T>(name);
  vop->set_value(val);
  value_ptr=vop->value();
  flx_data->write_message(FLX_DATADDDEBUG,cur_function,string("Creating object of type ")+typeid(T).name()+" named '"+name+"' at address "+flx_convert_to_string(value_ptr));
  return value_ptr;
}

/*****************************************************************************/

template <class T>
FlxObject *flx_create_scalar_object(const string &name,T *value){
  FlxScalarObject<T> *fop;
  
  fop=new FlxScalarObject<T>(name);
  fop->set_value(value);
  return fop;
}

/*****************************************************************************/

template <class T>
bool flx_do_object_convert(const string &s,T *&xp,const string &type_name){
  string cur_function="flx_do_object_convert";
  T *tempxp;

  flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Converting '"+s+"' to a "+type_name);

  // try to find a variable of the appropriate type
  if((tempxp=flx_get_object_by_name<T>(s))){
    if(xp){
      *xp=*tempxp;
    } else {
      xp=tempxp;
    }
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Converted as a variable");
    return true;
  } else {
    return false;
  }

} /* flx_do_object_convert */

/*****************************************************************************/

template <class T>
bool flx_object_convert(const string &s,T *&xp,const string &type_name){
  string cur_function="flx_object_convert";
  string clean_s;

  // strip off any leading $
  if(s.size() && s[0]=='$'){ 
    clean_s=s.substr(1);
  } else {
    clean_s=s;
  }

  if(flx_do_object_convert<T>(clean_s,xp,type_name)){
    return true;
  } else {
    // can't find an appropriate variable
    flx_data->write_message(FLX_DATAERROR,cur_function,"There is no "+type_name+" variable named '"+clean_s+"'");
    return false;
  }

} /* flx_object_convert */

/*****************************************************************************/

template <class T>
bool flx_object_convert_with_implicit(const string &s,T *&xp,const string &type_name){
  string cur_function="flx_object_convert_implicit";
  string clean_s;
  T *tempxp;
  
  // strip off any leading $
  if(s.size() && s[0]=='$'){ 
    clean_s=s.substr(1);
  } else {
    clean_s=s;
  }

  if(flx_do_object_convert<T>(clean_s,xp,type_name)){
    return true;
  // try an implicit conversion from a string
  } else if((tempxp=flx_implicit_object_conversion<T>(clean_s))){
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Did implicit conversion from a string variable");
    if(xp){
      *xp=*tempxp;
    } else {
      xp=tempxp;
    }
    return true;
  } else {
    // can't find an appropriate variable
    flx_data->write_message(FLX_DATAERROR,cur_function,"There is no "+type_name+" variable named '"+clean_s+"', and can't find a string variable to convert from");
    return false;
  }

} /* flx_object_convert_with_implicit */

/*****************************************************************************/

FLX_LINKAGE extern void flx_process_dependencies(FlxObject *object);

FLX_LINKAGE extern FlxObject *flx_dependency_root;

/*****************************************************************************/

#endif
