//LIC// FLXLab v2.5 - A program for running psychology experiments.  
//LIC// Copyright (C) 2010 Todd R. Haskell (todd.haskell@wwu.edu) 
//LIC// 
//LIC// Use and distribution is governed by the terms of the 
//LIC// GNU General Public License. Certain portions of the 
//LIC// program may be subject to other licenses as well. See 
//LIC// the file LICENSE.TXT for details.
//LIC// 
#include <string>
#include <iostream>
#include <fstream>
#include <iomanip>
#include "flxbase\FlxDataQueue.h"
#include "flxbase\FlxDataSystem.h"
#include "flxbase\file_system.h"
#include "flxbase\time.h"
#include "flxbase\FlxSetting.h"
#include "flxbase\string_utils.h"
#include "flxbase\flxbase.h"

using namespace std;

string FlxDataQueue::level_labels[]={"ERROR","DATA","INFO","EVENT","SCRIPT","HOOK","DEBUG","DDEBUG","DDDEBUG"};

/****************************************************************************/

void data_message_to_stderr(int level,const string &message,const string &source){

  cerr << message << " (" << source << ")" << endl;

} /* data_message_to_stderr */

/****************************************************************************/

FlxDataQueue::FlxDataQueue(const string &filename) {
				   
  d_filename=filename;
  d_stream=NULL;
  d_file_is_open=false;
  d_file_is_frozen=false;
  
  d_append_to_existing_file=false;
  d_buffer_messages=true;
  d_do_time_stamp=false;

  d_record_to_file_flags[FLX_DATAERROR]=true;
  d_record_to_file_flags[FLX_DATADATA]=true;
  d_record_to_file_flags[FLX_DATAINFO]=false;
  d_record_to_file_flags[FLX_DATAEVENT]=false;
  d_record_to_file_flags[FLX_DATASCRIPT]=false;
  d_record_to_file_flags[FLX_DATAHOOK]=false;
  d_record_to_file_flags[FLX_DATADEBUG]=false;
  d_record_to_file_flags[FLX_DATADDEBUG]=false;
  d_record_to_file_flags[FLX_DATADDDEBUG]=false;
  
} /* FlxDataQueue::FlxDataQueue */

/****************************************************************************/

void FlxDataQueue::set_append_to_existing_file(bool append_to_existing_file){
  flx_change_global_setting(&d_append_to_existing_file,append_to_existing_file,d_filename+"::append to existing file"); 
}

/****************************************************************************/

void FlxDataQueue::set_buffer_messages(bool buffer_messages){ 
  flx_change_global_setting(&d_buffer_messages,buffer_messages,d_filename+"::set_buffer_messages"); 
}

/****************************************************************************/

void FlxDataQueue::set_do_time_stamp(bool do_time_stamp){ 
  flx_change_global_setting(&d_do_time_stamp,do_time_stamp,d_filename+"::do_time_stamp"); 
}

/****************************************************************************/

void FlxDataQueue::set_record_to_file_flag(int level,bool flag){
  flx_change_global_setting(&d_record_to_file_flags[level],flag,d_filename+"::record_to_file_flag for level "+flx_convert_to_string(level)); 
}

/****************************************************************************/

void FlxDataQueue::open_file(void){
  string cur_function="FlxDataQueue::open_file";
  string expanded_filename;
  bool cur_buffering_flag;

  if(d_file_is_open){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Attempt to open data file when file is already open");
    return;
  }

  cur_buffering_flag=d_buffer_messages;
  d_buffer_messages=true; /* this ensures that any calls to
			   <write_message> during the process of
			   opening the data file won't trigger another
			   attempt to open the data file */
  d_stream=new ofstream((expanded_filename=flx_expand_new_filename(d_filename)).c_str(),(d_append_to_existing_file ? ios::app : ios::out));
  if(!*d_stream){
    d_stream=static_cast<ofstream*>(&cerr);
    write_message(FLX_DATAERROR,cur_function,"Can't open data file '"+expanded_filename+"', writing messages to console instead.");
  } else {
    write_message(FLX_DATADEBUG,cur_function,"Opening data file '"+expanded_filename+"'");
  }
  d_file_is_open=true;
  d_buffer_messages=cur_buffering_flag;

} /* FlxDataQueue::open_file */

/*****************************************************************************/

void FlxDataQueue::close_file(void){
  string cur_function="FlxDataQueue::close_file";

  write_message(FLX_DATADEBUG,cur_function,"Preparing to close data file");
  this->flush();

  /* a data file will only be opened if there are messages to write to it,
     so we can't assume that the file is open */
  if(d_file_is_open && d_stream!=&cerr){
    d_stream->close();
    delete d_stream;
    d_file_is_open=false;
  }

} /* FlxDataQueue::close_file */

/*****************************************************************************/

void FlxDataQueue::write_message(int level,const string &source,const string &message){
  string prefix;
  string suffix;
  long long time;

  switch(level){
  case FLX_DATADATA:
    prefix="";
    suffix="";
    break;
  case FLX_DATAERROR:
    prefix="!!!! ";
    suffix=" ("+source+")";
    break;
  case FLX_DATAINFO:
      prefix="** ";
      suffix="";
      break;
  case FLX_DATAEVENT:
    prefix="EVENT ";
    suffix="";
    break;
  case FLX_DATASCRIPT:
    prefix="SCRIPT ";
    suffix="";
    break;
  case FLX_DATAHOOK:
    prefix="  HOOK ";
    suffix="";
    break;
  case FLX_DATADEBUG:
    prefix="   % ";
    suffix="";
    break;
  case FLX_DATADDEBUG:
    prefix="      + ";
    suffix="";
    break;
  case FLX_DATADDDEBUG:
    prefix="         @ ";
    suffix="";
    break;
  }
  time=flx_get_time();
  d_queue.insert(new FlxDataMessage(level,time,prefix+message+suffix));
  if(flx_data->handler(level)){
    flx_data->handler(level)(level,message,source);
  }
  if(flx_data->abort_flag(level)){
    abort_flxlab=true;
  }
  if(!d_buffer_messages) flush();

} /* FlxDataQueue::write_message */

/*****************************************************************************/

bool FlxDataQueue::flush(void){
  string cur_function="FlxDataQueue::flush";
  FlxDataMessage *dm;

  if(!d_file_is_frozen){
    if(d_buffer_messages) write_message(FLX_DATAHOOK,cur_function,"Flushing data queue");    
    while(d_queue.read()){
      dm=d_queue.remove();    
      if(d_record_to_file_flags[dm->level()]){
	
	/* it would be more efficient to put this at the beginning of this
	   function, but by putting it here we ensure that a data file will
	   only be created if there are actually messages to write to it */
	if(!d_file_is_open){
	  open_file();
	}
    
	if(d_do_time_stamp){
	  *d_stream << setw(10) << setfill('0') << dm->time() << " ";
	}
	*d_stream << dm->message() << endl;
      }
      delete dm;
    }
  }
  return true;

} /* FlxDataQueue::flush */

/*****************************************************************************/


