//LIC// FLXLab v2.5 - A program for running psychology experiments.  
//LIC// Copyright (C) 2010 Todd R. Haskell (todd.haskell@wwu.edu) 
//LIC// 
//LIC// Use and distribution is governed by the terms of the 
//LIC// GNU General Public License. Certain portions of the 
//LIC// program may be subject to other licenses as well. See 
//LIC// the file LICENSE.TXT for details.
//LIC// 

#include <fstream>
#include <sstream>
#include "flxbase\FlxStimulusList.h"
#include "flxbase\logical_and_numeric_types.h"
#include "flxbase\strings.h"
#include "flxbase\scripts.h"
#include "flxbase\command_manager.h"
#include "flxbase\FlxDataSystem.h"
#include "flxbase\event_hooks.h"
#include "flxbase\condition_manager.h"
#include "flxbase\FlxObject.h"
#include "flxbase\FlxTriplexList.h"
#include "flxbase\scopes.h"
#include "flxbase\file_system.h"
#include "flxbase\random.h"
#include "flxbase\list_manager.h"

/* We need this so we don't have to qualify all STL names with
"std::" */
using namespace std;

FlxStimulusList *flx_cur_list;

/*****************************************************************************/

FlxStimulusList::~FlxStimulusList(){
  int i;
  
  if(d_items){
    for(i=0;i<d_num_items;i++) delete [] d_items[i];
    delete [] d_items;
    delete [] d_cur_item;
  }

} /* FlxStimulusList::~FlxStimulusList */

/*****************************************************************************/

bool get_field(istringstream &line_stream,string &next_field){
  string temp;

  if(line_stream >> next_field && !line_stream.fail()){
    // check for quoted fields
    if(next_field[0]=='"'){
      // check to make sure we don't have a single quoted word
      if(!(next_field.size()>1 && next_field[next_field.size()-1]=='"')){
	// get the rest of the field
	getline(line_stream,temp,'"');
	next_field+=(temp+"\""); /* need to replace the quote
				    that getline strips off */
      }
      next_field=flx_string_strip_quotes(next_field);
    }
    return true;
  } else {
    return false;
  }
  
} /* get_field */

/*****************************************************************************/

bool FlxStimulusList::read_from_stream(ifstream &input_stream){
  string cur_function="FlxStimulusList::read_from_stream";
  string next_line, next_field;
  FlxLinkedList<string> lines_list;
  istringstream *line_stream;
  int cur_line, cur_field;
  
  // read in all the lines from the file
  d_num_items=0;
  while(getline(input_stream,next_line) && !input_stream.fail()){
    if(!next_line.empty()){
      lines_list.insert(next_line);
      d_num_items++;
    }
  }
  
  // check that we actually read some lines
  if(d_num_items==0){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Empty stimulus list");
    return false;
  }
  
  // look at the first line to determine how many fields there are
  line_stream=new istringstream((lines_list.item())->c_str());
  d_num_fields=0;
  while(get_field(*line_stream,next_field)) d_num_fields++;
  delete line_stream;

  /* allocate storage for the list; we allocate storage for individual fields
     on the fly (see below) */
  d_items=new string *[d_num_items];
  d_cur_item=new string[d_num_fields];

  // go back through all the lines and parse them into fields
  cur_line=0;
  while(lines_list.item()){
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Parsing line "+flx_convert_to_string(cur_line));
    
    // allocate storage for the fields
    d_items[cur_line]=new string[d_num_fields];
    
    // parse the line
    next_line=*(lines_list.item());
    line_stream=new istringstream(next_line.c_str());
    cur_field=0;
    while(get_field(*line_stream,next_field)){
      flx_data->write_message(FLX_DATADDDEBUG,cur_function,next_field);
      if(cur_field<d_num_fields) d_items[cur_line][cur_field]=next_field;
      cur_field++;
    }
    delete line_stream;

    // check that we got the right number of fields
    if(cur_field!=d_num_fields){
      // we've got a problem, free the memory allocated so far and abort
      while(cur_line>=0){
	delete [] d_items[cur_line];
	cur_line--;
      }
      delete [] d_items;
      delete [] d_cur_item;
      d_items=NULL;
      d_cur_item=NULL;
      flx_data->write_message(FLX_DATAERROR,cur_function,"Not all lines in the stimulus list have the same number of fields (possible blank line)");
      return false;
    }

    lines_list.remove();
    cur_line++;
  }

  d_end_of_list=false;
  d_item_index=-1; /* This indicates that we're ready to start reading items
		    from the list */
  return true;
	
} /* FlxStimulusList::read_from_stream */

/*****************************************************************************/

bool FlxStimulusList::join_lists(FlxLinkedList<FlxStimulusList *> &the_lists){
  string cur_function="FlxStimulusList::join_lists";
  FlxStimulusList **slh;
  long i, j;
  string **item_ptr;

  /* Determine the total number of items, and make sure that all the lists
  have the same number of fields */
  d_num_items=0;
  d_num_fields=0;
  while((slh=the_lists.item())){
	d_num_items+=(*slh)->num_items();
	if(d_num_fields==0){
		d_num_fields=(*slh)->num_fields();
	} else if(d_num_fields!=(*slh)->num_fields()){
      flx_data->write_message(FLX_DATAERROR,cur_function,"Can't join stimulus lists, the lists do not all have the same number of fields");
		return false;
	}
	the_lists.advance();
  }
  the_lists.rewind();

  /* Allocate storage for the list; we allocate storage for individual fields
     on the fly (see below) */
  d_items=new string *[d_num_items];
  d_cur_item=new string[d_num_fields];

  /* Go back through each list, and copy all the fields into the combined list */
	item_ptr=d_items;
  while((slh=the_lists.item())){
		for(i=0;i<(*slh)->num_items();i++){
			*item_ptr=new string[d_num_fields];
			(*slh)->update();
			for(j=0;j<d_num_fields;j++){
				(*item_ptr)[j]=*((*slh)->field_pointer(j));
			}
			item_ptr++;
		}
		(*slh)->reset();
		the_lists.advance();
  }
  the_lists.rewind();

  d_end_of_list=false;
  d_item_index=-1; /* This indicates that we're ready to start reading items
		    from the list */
  return true;

} /* FlxStimulusList::join_lists */

/*****************************************************************************/

/* This uses the Fisher-Yates shuffle algorithm */

void FlxStimulusList::randomize(void){
	string cur_function="FlxStimulusList::randomize";
	long i, j;
	string *p;
	
	if(d_num_items>1){
		for(i=0;i<d_num_items-1;i++){
			j=flx_get_random(i+1,d_num_items-1);
			flx_data->write_message(FLX_DATAINFO,cur_function,"i="+flx_convert_to_string(i)+" j="+flx_convert_to_string(j));
			p=d_items[j];
			d_items[j]=d_items[i];
			d_items[i]=p;
		}
	}

} /* FlxStimulusList::randomize */

/*****************************************************************************/

bool FlxStimulusList::update(void){
  string cur_function="FlxStimulusList::update";
  int i;

  if(d_end_of_list){
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"At end of stimulus list '"+d_name+"'");
    return false;
  } else {
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Advancing to next item in stimulus list '"+d_name+"'");
    d_item_index++;
    if(d_item_index>=d_num_items-1){
      d_end_of_list=true;
    }
    for(i=0;i<d_num_fields;i++){
      d_cur_item[i]=d_items[d_item_index][i];
    }
    return true;
  }

} /* FlxStimulusList::update */

/*****************************************************************************/

void FlxStimulusList::add_to_update_list(FlxLinkedList<FlxObject *> &update_list){
  
  if(flx_cur_list==this) FlxObject::add_to_update_list(update_list); else flx_data->write_message(FLX_DATADDEBUG,"FlxStimulusList::add_to_update_list","'"+d_name+"' is not the current list, so it won't be updated"); 

} /* FlxStimulusList::add_to_update_list */

/*****************************************************************************/

string *FlxStimulusList::field_pointer(int field_num){

  if(field_num<0 || field_num>=d_num_fields){
    return NULL;
  } else {
    return &(d_cur_item[field_num]);
  }

} /* FlxStimulusList::field_pointer */

/*****************************************************************************/
