//LIC// FLXLab v2.5 - A program for running psychology experiments.  
//LIC// Copyright (C) 2010 Todd R. Haskell (todd.haskell@wwu.edu) 
//LIC// 
//LIC// Use and distribution is governed by the terms of the 
//LIC// GNU General Public License. Certain portions of the 
//LIC// program may be subject to other licenses as well. See 
//LIC// the file LICENSE.TXT for details.
//LIC// 
#include "flxbase\linkage.h"
#include "flxbase\FlxDataSystem.h"
#include "flxbase\FlxDuplexList.h"
#include "flxbase\OS_specific_file_system.h"
#include "flxbase\string_utils.h"

/* For purposes of working with the file system, we treat OS X in the
   same way as Linux */
#ifdef FLX_MAC
#define FLX_LINUX
#endif

#ifdef FLX_LINUX
#include <sys\types.h>
#include <sys\stat.h>
#include <sys\wait.h>
#include <dirent.h>
#include <unistd.h>
#define FLX_MAX_PATH_SIZE 1000
FLX_LINKAGE string flx_path_separator="/";
#endif

#ifdef FLX_WINDOWS
#include <io.h>
#include <windows.h>
#include <sys\types.h>
#include <sys\stat.h>
#include <direct.h>
FLX_LINKAGE string flx_path_separator="\\";
#endif

extern FlxDuplexList<string,string> search_directory_list;

/*****************************************************************************/

FLX_LINKAGE string flx_current_working_directory(void){
  string cur_function="flx_current_working_directory";

  const int max_directory_length=200;
  char buffer[max_directory_length];
  string cwd;

#ifdef FLX_LINUX
  if(getcwd(buffer,max_directory_length)){
#endif
#ifdef FLX_WINDOWS
	  if(_getcwd(buffer,max_directory_length)){
#endif
    cwd=buffer;
    // make sure the directory name ends with a path separator
    if(cwd.size() && cwd.substr(cwd.size()-1,1)!=flx_path_separator){
      cwd=cwd+flx_path_separator;
    }
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Current working directory is '"+cwd+"'");
    return cwd;
  } else {
    flx_data->write_message(FLX_DATAERROR,cur_function,"Path of current working directory is too long, using null string instead");
    return string();
  }

} /* flx_current_working_directory */

/*****************************************************************************/

FLX_LINKAGE bool flx_is_absolute_path(const string &file_path){

#ifdef FLX_LINUX
  return (file_path.size() && (file_path.substr(0,1)==flx_path_separator));
#endif

#ifdef FLX_WINDOWS
  return ((file_path.size()>1) && (file_path.substr(1,1)==":"));
#endif

} /* flx_is_absolute_path */

/*****************************************************************************/

FLX_LINKAGE string flx_expand_filename(const string &file_name){
	string cur_function="flx_expand_filename";
#ifdef FLX_LINUX
  struct stat *sp;
#endif
#ifdef FLX_WINDOWS
  struct _stat *sp;
#endif
  string *search_path, expanded_file_name;
  int result=1;
  
  flx_data->write_message(FLX_DATADEBUG,cur_function,"Expanding file name '"+file_name+"'");
#ifdef FLX_LINUX
  sp = new struct stat();
#endif
#ifdef FLX_WINDOWS
  sp = new struct _stat();
#endif

  /* if the file name is an absolute path, just check if it exists */
  if(flx_is_absolute_path(file_name)){
    expanded_file_name=file_name;
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Checking for '"+expanded_file_name+"'");
#ifdef FLX_LINUX
    result=stat(expanded_file_name.c_str(),sp);
#endif
#ifdef FLX_WINDOWS
    result=_stat(expanded_file_name.c_str(),sp);
#endif
  } else {
      /* search for the file in each path in the search path list */
      while((search_path=search_directory_list.item())){
	expanded_file_name=*search_path+file_name;
	flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Checking for '"+expanded_file_name+"'");
#ifdef FLX_LINUX
	result=stat(expanded_file_name.c_str(),sp);
#endif
#ifdef FLX_WINDOWS
	result=_stat(expanded_file_name.c_str(),sp);
#endif
	if(!result){
	  break;
	}
	search_directory_list.advance();
      }
      search_directory_list.rewind();
  }
  delete sp;
  
  if(result){
    return string();
  } else {
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"The file exists, returning this name as the expansion");
    return expanded_file_name;
  }

 } /* flx_expand_filename */

/*****************************************************************************/

FLX_LINKAGE void flx_get_directory_entries(const string &dir_path,FlxLinkedList<string> &file_list,int type_flag){
  string cur_function="flx_get_directory_entries";

  flx_data->write_message(FLX_DATADEBUG,cur_function,"Getting directory entries for '"+dir_path+"'");
#ifdef FLX_LINUX
  DIR *dir_stream;
  struct dirent *directory_entry_info;
  struct stat file_info;

  if((dir_stream=opendir(dir_path.c_str()))){
    while((directory_entry_info=readdir(dir_stream))){
      if(string(directory_entry_info->d_name)!="."){
	if(!stat((dir_path+flx_path_separator+string(directory_entry_info->d_name)).c_str(),&file_info)){
	  if( ((type_flag==FLX_FS_FILES) && (!S_ISDIR(file_info.st_mode))) || ((type_flag==FLX_FS_DIRECTORIES) && (S_ISDIR(file_info.st_mode))) || (type_flag==FLX_FS_ALL) ){
		flx_data->write_message(FLX_DATADDDEBUG,cur_function,directory_entry_info->d_name);
		file_list.insert_sorted(directory_entry_info->d_name);
	  }
	} else {
	  flx_data->write_message(FLX_DATAERROR,cur_function,"Can't get info for directory entry '"+string(directory_entry_info->d_name)+"', skipping");
	}
      }
    }
    closedir(dir_stream);
	file_list.rewind();
  } else {
    flx_data->write_message(FLX_DATAERROR,cur_function,"Can't open directory '"+dir_path+"'");
  }
#endif
#ifdef FLX_WINDOWS
	struct _finddata_t file_info;
	intptr_t search_ptr;
	int result;

	if((result=search_ptr=_findfirst((dir_path+"\\*").c_str(),&file_info))!=-1){
		while(result!=-1){
			if(string(file_info.name)!="."){
				if( ((type_flag==FLX_FS_FILES) && (!(file_info.attrib&_A_SUBDIR))) || ((type_flag==FLX_FS_DIRECTORIES) && (file_info.attrib&_A_SUBDIR)) || (type_flag==FLX_FS_ALL)){
						flx_data->write_message(FLX_DATADDDEBUG,cur_function,file_info.name);
						file_list.insert_sorted(string(file_info.name));
				}
			}
			result=_findnext(search_ptr,&file_info);
		}
		_findclose(search_ptr);
		file_list.rewind();
	} else {
		flx_data->write_message(FLX_DATAERROR,cur_function,"Directory '"+dir_path+"' does not exist or is empty");
	}	
#endif

} /* flx_get_directory_entries */

/*****************************************************************************/

FLX_LINKAGE bool flx_run_command(const string &command,const string &argument){

#ifdef FLX_LINUX
  
    system((command+" "+argument).c_str());
    return true;

#endif

#ifdef FLX_WINDOWS
	STARTUPINFO si;
	PROCESS_INFORMATION pi;
	string command_line="\""+command+"\" \""+argument+"\"";

	ZeroMemory( &si, sizeof(si) );
	si.cb = sizeof(si);
	ZeroMemory( &pi, sizeof(pi) );
	if(!CreateProcess(NULL,const_cast<char *>(command_line.c_str()),NULL,NULL,FALSE,0,NULL,NULL,&si,&pi)) return false;
	WaitForSingleObject(pi.hProcess,INFINITE);
	CloseHandle( pi.hProcess );
	CloseHandle( pi.hThread );
	return true;
#endif

} /* flx_run_command */

/*****************************************************************************/

