//LIC// FLXLab v2.5 - A program for running psychology experiments.  
//LIC// Copyright (C) 2010 Todd R. Haskell (todd.haskell@wwu.edu) 
//LIC// 
//LIC// Use and distribution is governed by the terms of the 
//LIC// GNU General Public License. Certain portions of the 
//LIC// program may be subject to other licenses as well. See 
//LIC// the file LICENSE.TXT for details.
//LIC// 

/* For purposes of working with time, we treat OS X the same as Linux */
#ifdef FLX_MAC
#define FLX_LINUX
#endif

#ifdef FLX_LINUX
#include <sys\time.h>
#include <sys\select.h>
#endif

#ifdef FLX_WINDOWS
#include <windows.h>

LARGE_INTEGER counter_frequency;
HANDLE current_process;
#endif

#include <string>
#include "flxbase\linkage.h"

using namespace std;

#define FLX_NO_TIME_UNITS 0
#define FLX_MILLISECONDS 1
#define FLX_MICROSECONDS 2

FLX_LINKAGE bool flx_microseconds_enabled;
FLX_LINKAGE bool flx_using_microseconds;

/* Note that for microseconds in Windows this is the value of the high-performance counter,
 for Linux it is the actual number of microseconds */
long long base_time;

/* When we have switched from milli- to microseconds or vice versa, we
   need to reset the base time to 0, or else the values will be
   difficult to interpret.  We use <last_time> to keep track of what
   units we were using as of the last call to get the time, and reset
   the base time whenever this variable changes. */
int last_time=FLX_NO_TIME_UNITS;

/*****************************************************************************/

FLX_LINKAGE long long flx_get_milliseconds(void){

#ifdef FLX_LINUX
  timeval tv;

  gettimeofday(&tv,NULL);
  if(last_time!=FLX_MILLISECONDS){
	  base_time=(tv.tv_sec & 017777)*1000 + tv.tv_usec/1000;
	  last_time=FLX_MILLISECONDS;
	  return 0;
  } else {
	return (tv.tv_sec & 017777)*1000 + tv.tv_usec/1000 - base_time;
  }
#endif
#ifdef FLX_WINDOWS
  if(last_time!=FLX_MILLISECONDS){
	  base_time=timeGetTime();
	  last_time=FLX_MILLISECONDS;
	  return 0;
  } else {
	return timeGetTime() - base_time;
  }
#endif

} /* flx_get_milliseconds */

/*****************************************************************************/

FLX_LINKAGE long long flx_get_microseconds(void){
#ifdef FLX_LINUX
  timeval tv;

  gettimeofday(&tv,NULL);
  if(last_time!=FLX_MICROSECONDS){
	  base_time=(tv.tv_sec & 017777)*1000000 + tv.tv_usec;
	  last_time=FLX_MICROSECONDS;
	  return 0;
  } else {
	return ((tv.tv_sec & 017777))*1000000 + tv.tv_usec - base_time;
  }
#endif
#ifdef FLX_WINDOWS
	LARGE_INTEGER current_counter;

	if(flx_microseconds_enabled){
		QueryPerformanceCounter(&current_counter);
		if(last_time!=FLX_MICROSECONDS){
			base_time=current_counter.QuadPart;
			last_time=FLX_MICROSECONDS;
			return 0;
		} else {
			return (current_counter.QuadPart-base_time)*1000000/counter_frequency.QuadPart;
		}
	} else { // shouldn't need this, but just in case...
		// if we don't have a microsecond timer, emulate it using the millisecond timer
		return flx_get_milliseconds()*1000;
	}
#endif

} /* flx_get_microseconds */

/*****************************************************************************/

void flx_init_timer(void){

#ifdef FLX_WINDOWS
	timeBeginPeriod(1);
	if(QueryPerformanceFrequency(&counter_frequency)){
		flx_microseconds_enabled=true;
	} else {
		flx_microseconds_enabled=false;
	}
	current_process=GetCurrentProcess();
#endif

#ifdef FLX_LINUX
	flx_microseconds_enabled=true;
#endif

  flx_using_microseconds=false; // use milliseconds by default
  flx_get_milliseconds(); // reset the timer

} /* flx_init_timer */

/*****************************************************************************/

void flx_exit_timer(void){
#ifdef FLX_WINDOWS
	timeEndPeriod(1);
#endif

} /* flx_exit_timer */

/*****************************************************************************/
