//LIC// FLXLab v2.5 - A program for running psychology experiments.  
//LIC// Copyright (C) 2010 Todd R. Haskell (todd.haskell@wwu.edu) 
//LIC// 
//LIC// Use and distribution is governed by the terms of the 
//LIC// GNU General Public License. Certain portions of the 
//LIC// program may be subject to other licenses as well. See 
//LIC// the file LICENSE.TXT for details.
//LIC// 
#include <sstream>
#include "flxbase\FlxDataSystem.h"
#include "flxbase\scripts.h"
#include "flxbase\FlxTriplexList.h"
#include "flxbase\event_manager.h"
#include "flxbase\event_hooks.h"
#include "flxbase\command_manager.h"
#include "flxbase\strings.h"
#include "flxbase\logical_and_numeric_types.h"
#include "flxbase\string_utils.h"
#include "flxbase\condition_manager.h"
#include "flxbase\object_manager.h"
#include "flxbase\linkage.h"

/*****************************************************************************/

#define FLX_CONDITION_REPEAT_INDEFINITELY -1

class RepeatCondition : public FlxCondition {
  long *d_max_repetitions, d_cur_repetitions;
public:
  RepeatCondition(const string &name,long *max_repetitions) : FlxCondition(name), d_max_repetitions(max_repetitions), d_cur_repetitions(0) {}
  ~RepeatCondition(void) {}
  bool evaluate(void){ 
    if(*d_max_repetitions==FLX_CONDITION_REPEAT_INDEFINITELY){
      return true;
    } else if(d_cur_repetitions<*d_max_repetitions){ 
     d_cur_repetitions++; 
      return true;
    } else {
      return false;
    }
  }
  void reset(void) { d_cur_repetitions=0; }
};

/*****************************************************************************/
  
FlxCondition *repeat_condition_parser(const string &keyword,int arg_count,string *arg_array){
  string cur_function="repeat_condition_parser";
  string reps_string;
  long *max_reps=NULL;
  string name;
  FlxCondition *cp;

  if(arg_count>1){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": Bad repeat condition");
    return NULL;
  } else if(arg_count==0){
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Parsed repeat condition (indefinite)");
    max_reps=flx_long(FLX_CONDITION_REPEAT_INDEFINITELY);
  } else if(arg_count==1){
    reps_string=arg_array[0];
    flx_type_convert(reps_string,max_reps);
    if(max_reps){
      flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Parsed repeat condition (integer)");
    } else {
      flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": Can't convert '"+reps_string+"' to an integer in repeat condition");
      return NULL;
    }
  } 

  // actually create the condition
  name=flx_unique_name();
  flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Creating repeat condition named '"+name+"'");
  cp=new RepeatCondition(name,max_reps);
  flx_add_scalar_source(cp,max_reps);
  return cp;

} /* repeat_condition_parser */

/*****************************************************************************/

#define FLX_CONDITION_AND 1
#define FLX_CONDITION_OR 2

class CombinationCondition : public FlxCondition {
  FlxCondition *d_condition1, *d_condition2;
  int d_combination_type;
public:
  CombinationCondition(const string &name,FlxCondition *condition1,FlxCondition *condition2,int combination_type) : FlxCondition(name), d_condition1(condition1), d_condition2(condition2), d_combination_type(combination_type) {}
  ~CombinationCondition() {}
  bool evaluate(void);
  void reset(void){ d_condition1->reset(); d_condition2->reset(); }
};

bool CombinationCondition::evaluate(void){
  switch(d_combination_type){
  case FLX_CONDITION_AND:
    return d_condition1->evaluate() && d_condition2->evaluate();
    break;
  case FLX_CONDITION_OR:
    return d_condition1->evaluate() || d_condition2->evaluate();
    break;
  default:
    return false;
  }
}

/*****************************************************************************/

FlxCondition *combination_condition_parser(const string &keyword,int arg_count,string *arg_array){
  string cur_function="combination_condition_parser";
  string  combination_keyword, separator, condition1string="", condition2string="";
  int cur_arg=0, type;
  bool separator_found=false;
  FlxCondition *condition1=NULL, *condition2=NULL, *cp;
  string name;

  if(keyword=="both"){
    separator="and";
    type=FLX_CONDITION_AND;
  } else if(keyword=="either"){
    separator="or";
    type=FLX_CONDITION_OR;
  } else {
    flx_data->write_message(FLX_DATAERROR,cur_function,"Unknown combination condition keyword '"+keyword+"'");
    return NULL;
  }    
  
  while(cur_arg<arg_count){
    if(!separator_found && arg_array[cur_arg]==separator){
      separator_found=true;
    } else {
      if(!separator_found){
	condition1string+=(arg_array[cur_arg]+" ");
      } else {
	condition2string+=(arg_array[cur_arg]+" ");
      }
    }
    cur_arg++;
  }
  flx_type_convert(condition1string,condition1);
  if(!condition1){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Can't understand first condition in "+keyword+" combination");
    return NULL;
  }
  flx_type_convert(condition2string,condition2);
  if((condition1==NULL) || (condition2==NULL)){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Can't understand second condition in "+keyword+" combination");
    return NULL;
  }

  name=flx_unique_name();
  flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Creating new "+keyword+" condition named '"+name+"'");
  cp=new CombinationCondition(name,condition1,condition2,type);
  flx_add_object_source(cp,condition1);
  flx_add_object_source(cp,condition2);
  return cp;

} /* combination_condition_parser */

/*****************************************************************************/

class NotCondition : public FlxCondition {
  FlxCondition *d_condition;
public:
  NotCondition(const string &name,FlxCondition *condition) : FlxCondition(name), d_condition(condition) {}
  ~NotCondition() {}
  bool evaluate(void){ return !d_condition->evaluate(); }
  void reset(void){ d_condition->reset(); }
};

/*****************************************************************************/

FlxCondition *not_condition_parser(const string &keyword,int arg_count,string *arg_array){
  string cur_function="not_condition_parser";
  string condition_string="";
  int cur_arg=0;
  FlxCondition *condition=NULL, *cp;
  string name;

  /* parse out everything after the 'not' and convert it to the
     condition being negated */
  while(cur_arg<arg_count){
    condition_string+=(arg_array[cur_arg]+" ");
    cur_arg++;
  }
  flx_type_convert(condition_string,condition);
  if(!condition){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Bad not condition");
    return NULL;
  }

  // create the 'not' condition
  name=flx_unique_name();
  flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Creating new not condition named '"+name+"'");
  cp=new NotCondition(name,condition);
  flx_add_object_source(cp,condition);
  return cp;
}

/*****************************************************************************/

#define FLX_CONDITION_UNTIL 1
#define FLX_CONDITION_WHEN 2
#define FLX_CONDITION_AFTER 3
#define FLX_CONDITION_WHENEVER 4

class UntilWhenAfterWheneverCondition : public FlxCondition {
  FlxCondition *d_condition;
  int d_condition_type;
  bool d_condition_triggered;
public:
  UntilWhenAfterWheneverCondition(const string &name,FlxCondition *condition,int condition_type) : FlxCondition(name), d_condition(condition), d_condition_type(condition_type), d_condition_triggered(false) {}
  ~UntilWhenAfterWheneverCondition() {}
  bool evaluate(void){
    switch(d_condition_type){
    case FLX_CONDITION_WHEN: case FLX_CONDITION_WHENEVER:
      d_condition_triggered=d_condition->evaluate();
      if(d_condition_triggered && d_condition_type==FLX_CONDITION_WHENEVER) d_condition->reset();
      return d_condition_triggered;
      break;
    case FLX_CONDITION_AFTER: case FLX_CONDITION_UNTIL:
      if(!d_condition_triggered && d_condition->evaluate()) d_condition_triggered=true;
      return (d_condition_type==FLX_CONDITION_UNTIL) ? !d_condition_triggered : d_condition_triggered;
      break;
    default:
      return false;
      break;
    }
  }
  void reset(void){ d_condition_triggered=false; d_condition->reset(); }
};

/*****************************************************************************/

FlxCondition *modifier_condition_parser(const string &keyword,int arg_count,string *arg_array){
  string cur_function="modifier_condition_parser";
  string condition_string="";
  int cur_arg=0;
  FlxCondition *modified_condition=NULL, *modifier_condition;
  string name;


  /* parse out everything after the keyword and convert it to the
     condition being modified */
  while(cur_arg<arg_count){
    condition_string+=(arg_array[cur_arg]+" ");
    cur_arg++;
  }
  flx_type_convert(condition_string,modified_condition);
  if(!modified_condition){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Bad "+keyword+" condition");
    return NULL;
  } 

  // create the modifier condition
  name=flx_unique_name();
  if(keyword=="until"){
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Parsed until condition");
    modifier_condition=new UntilWhenAfterWheneverCondition(name,modified_condition,FLX_CONDITION_UNTIL);
  } else if(keyword=="after"){
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Parsed after condition");
    modifier_condition=new UntilWhenAfterWheneverCondition(name,modified_condition,FLX_CONDITION_AFTER);
  } else if(keyword=="when"){
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Parsed when condition");
    modifier_condition=new UntilWhenAfterWheneverCondition(name,modified_condition,FLX_CONDITION_WHEN);
  } else if(keyword=="whenever"){
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Parsed whenever condition");
    modifier_condition=new UntilWhenAfterWheneverCondition(name,modified_condition,FLX_CONDITION_WHENEVER);
  } else {
    flx_data->write_message(FLX_DATAERROR,cur_function,"Modifier condition parser called with unrecognized condition keyword '"+keyword+"'");
    return NULL;
  }
  flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Creating modifier condition named '"+name+"'");
  flx_add_object_source(modifier_condition,modified_condition);
  return modifier_condition;

} /* modifier_condition_parser */

/*****************************************************************************/

void flx_basic_conditions_init(void){

  flx_add_condition_type("repeat",repeat_condition_parser);
  flx_add_condition_type("both",combination_condition_parser);
  flx_add_condition_type("either",combination_condition_parser);
  flx_add_condition_type("not",not_condition_parser);
  flx_add_condition_type("until",modifier_condition_parser);
  flx_add_condition_type("after",modifier_condition_parser);
  flx_add_condition_type("when",modifier_condition_parser);
  flx_add_condition_type("whenever",modifier_condition_parser);

} /* flx_basic_conditions_init */

/*****************************************************************************/
