//LIC// FLXLab v2.5 - A program for running psychology experiments.  
//LIC// Copyright (C) 2010 Todd R. Haskell (todd.haskell@wwu.edu) 
//LIC// 
//LIC// Use and distribution is governed by the terms of the 
//LIC// GNU General Public License. Certain portions of the 
//LIC// program may be subject to other licenses as well. See 
//LIC// the file LICENSE.TXT for details.
//LIC// 
#include <typeinfo>
#include "flxbase\FlxDataSystem.h"
#include "flxbase\logical_and_numeric_types.h"
#include "flxbase\command_manager.h"
#include "flxbase\scripts.h"
#include "flxbase\scopes.h"
#include "flxbase\FlxLinkedList.h"
#include "flxbase\FlxStack.h"
#include "flxbase\time.h"
#include "flxbase\condition_manager.h"
#include "flxbase\strings.h"
#include "flxbase\event_hooks.h"
#include "flxbase\object_manager.h"
#include "flxbase\FlxGroupingEvent.h"
#include "flxbase\FlxSetting.h"
#include "flxbase\event_manager.h"

FLX_LINKAGE long flx_event_time, flx_data_time;

FLX_LINKAGE bool flx_force_exit;

/*****************************************************************************/

bool update_event_and_data_time(void){
 long long cur_time;

  cur_time=flx_get_time();
  flx_event_time=cur_time-event_start_time;
  flx_data_time=cur_time-data_start_time;
  return true;

} /* update_event_and_data_time */

/*****************************************************************************/

FLX_LINKAGE bool flx_type_convert(const string &s,FlxEvent *&ep){

  return flx_object_convert<FlxEvent>(s,ep,"FlxEvent");
} 

/*****************************************************************************/

bool NewGroupingEvent(string *name,FlxCondition *cp){
  string cur_function="NewGroupingEvent";
  FlxEvent *ep;
  
  if(flx_cur_script_command=="BlockEvent"){
    ep=new BlockEvent(*name,cp);
    flx_default_condition_text="repeat";
  } else if(flx_cur_script_command=="TrialEvent"){
    ep=new TrialEvent(*name,cp);
    flx_default_condition_text="repeat 1";
  } else if(flx_cur_script_command=="ExperimentEvent" || flx_cur_script_command=="GroupingEvent"){
    ep=new ExperimentEvent(*name,cp);
    flx_default_condition_text="repeat 1";
  } else {
    flx_data->write_message(FLX_DATAERROR,cur_function,"NewGroupingEvent called via unknown name "+flx_cur_script_command);  
    return false;
  }
  flx_data->write_message(FLX_DATASCRIPT,cur_function,"Creating new "+flx_cur_script_command+" '"+*name+"'");
  flx_add_object_source(ep,cp);
  return true;

} /* NewGroupingEvent */

/*****************************************************************************/

bool SelectEvent(string *raw_argument){
/* Because the special event 'default' should have all possible event
   properties, we can't implement it as an actual object, or modules
   wouldn't be able to add new event properties without changing
   the definition of the default object. Thus, the default object is
   implemented as a null pointer. Any routine that sets event
   properties then knows that if <flx_cur_event> is null, it
   should change the default value. The down side of this is that we
   can't pass in a null <Event *> to <SelectEvent>, because
   the <flx_type_convert> call to convert the raw argument to an event
   would fail. Thus, we ask for the raw string argument and
   check if it is 'default'.  If so, we set <flx_cur_event> to
   null, otherwise we convert to a <Event *> and assign the
   result to <flx_cur_event>. */
  string cur_function="SelectEvent";
  FlxEvent *ep=NULL;
  TrialEvent *tep;
  BlockEvent *bep;
  ExperimentEvent *eep;

  if(*raw_argument=="default"){
    flx_data->write_message(FLX_DATASCRIPT,cur_function,"Selecting event 'default'");
    flx_change_global_setting<FlxEvent *>(&flx_cur_event,NULL,"flx_cur_event");
    return true;
  } else {
    if(flx_type_convert(*raw_argument,ep)){
	flx_data->write_message(FLX_DATASCRIPT,cur_function,"Selecting event '"+ep->name()+"'");
	flx_change_global_setting(&flx_cur_event,ep,"flx_cur_event");
	/* To understand the next few lines, consider the following sequence
           of events:
           1) Grouping event A is created
           2) Grouping event B is created
           3) Event A is selected
           4) Event B is added to event A
           Because the default trigger condition is set when a grouping
           event is created, this will cause the AddEvent command to use
           the condition corresponding to event B, rather than event A.
           To avoid this problem, we have to set the trigger condition
           not only when a grouping event is created, but also when
           it is selected. */
        if((tep=dynamic_cast<TrialEvent *>(flx_cur_event))){
          flx_default_condition_text="repeat 1";
        } else if((bep=dynamic_cast<BlockEvent *>(flx_cur_event))){
          flx_default_condition_text="repeat";
        } else if((eep=dynamic_cast<ExperimentEvent *>(flx_cur_event))){
          flx_default_condition_text="repeat 1";
        }
	return true;
    } else {
      flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": Argument is the wrong type, skipping command");
      return false;
    }
  }

} /* SelectEvent */

/*****************************************************************************/

bool AddEvent(FlxEvent *ep,FlxCondition *cp){
  string cur_function="AddEvent";
  GroupingEvent *gep;

  // first, check the current event to make sure it can have sub-events
  // syntax for dynamic casts: dynamic_cast<Derived *>(ep), with return value of 0 on failure
  if((gep=dynamic_cast<GroupingEvent *>(flx_cur_event))){
    flx_data->write_message(FLX_DATASCRIPT,cur_function,"Adding sub-event '"+ep->name()+"' to '"+flx_cur_event->name()+"'");
    gep->flx_add_event(ep,cp);
    flx_add_object_source(gep,ep);
    flx_add_object_source(gep,cp);
    return true;
  } else {
    flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": "+string(typeid(*flx_cur_event).name())+" can't have sub-events");
    return false;
  }
  
} /* AddEvent */

/*****************************************************************************/

bool ResetEventTime(bool *flag){
  string cur_function="ResetEventTime";

  flx_data->write_message(FLX_DATASCRIPT,cur_function,"Will reset the event time at '"+flx_cur_event->name()+"'");
  flx_cur_event->set_reset_event_time(flag);
  return true;

} /* ResetEventTime */

/*****************************************************************************/

bool ResetDataTime(bool *flag){
  string cur_function="ResetDataTime";

  flx_data->write_message(FLX_DATASCRIPT,cur_function,"Will reset the data time at '"+flx_cur_event->name()+"'");
  flx_cur_event->set_reset_data_time(flag);
  return true;

} /* ResetDataTime */

/*****************************************************************************/

bool Start(FlxEvent *ep){
  string cur_function="Start";

  flx_data->write_message(FLX_DATASCRIPT,cur_function,"Starting event '"+ep->name()+"'");
  ep->execute();
  flx_force_exit=false;
  return true;

} /* StartEvent */

/*****************************************************************************/

class EventCondition : public FlxCondition {
  bool *execution_flag;
  bool triggered;
public:
  EventCondition(const string &name,bool *flag) : FlxCondition(name), execution_flag(flag), triggered(false) {}
  ~EventCondition(void) {}
  bool evaluate(void);
  void reset(void) { triggered=false; }
};

bool EventCondition::evaluate(void){
  if(*execution_flag && !triggered){
    triggered=true;
    return true;
  } else {
    return false;
  }
}

/*****************************************************************************/

FlxCondition *event_condition_parser(const string &keyword,int arg_count,string *arg_array){
  string cur_function="event_condition_parser";
  FlxEvent *trigger_event=NULL;
  string name;
  FlxCondition *cp;

  if(arg_count!=1){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": Bad event condition");
    return NULL;
  } else {
    flx_type_convert(arg_array[0],trigger_event);
    if(trigger_event){
      name=flx_unique_name();
      flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Creating event condition named '"+name+"'");
      cp=new EventCondition(name,trigger_event->get_execution_flag());
      flx_add_object_source(cp,trigger_event);
      return cp;
    } else {
      flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": '"+arg_array[0]+"' isn't an event");
      return NULL;
    }
  }

} /* event_condition_parser */

/*****************************************************************************/

void flx_event_manager_init(void){

  flx_add_command("TrialEvent",NewGroupingEvent,"repeat 1");
  flx_add_command("ExperimentEvent",NewGroupingEvent,"repeat 1");
  flx_add_command("GroupingEvent",NewGroupingEvent,"repeat 1");
  flx_add_command("BlockEvent",NewGroupingEvent,"repeat 1");
  flx_add_command("ResetEventTime",ResetEventTime,"true");
  flx_add_command("ResetDataTime",ResetDataTime,"true");
  flx_add_command("Start",Start);
  flx_add_command("SelectEvent",SelectEvent);
  flx_add_command("AddEvent",AddEvent,"default_condition");
  flx_add_condition_type("event",event_condition_parser);
  flx_add_event_hook("update_event_and_data_time",update_event_and_data_time,FLX_EVENT_HOOK_WITHINEVENT);

  flx_event_time=flx_data_time=0;
  flx_force_exit=false;
  flx_cur_event=NULL;
  in_session=false;

} /* flx_event_manager_init */

/*****************************************************************************/

