//LIC// FLXLab v2.5 - A program for running psychology experiments.  
//LIC// Copyright (C) 2010 Todd R. Haskell (todd.haskell@wwu.edu) 
//LIC// 
//LIC// Use and distribution is governed by the terms of the 
//LIC// GNU General Public License. Certain portions of the 
//LIC// program may be subject to other licenses as well. See 
//LIC// the file LICENSE.TXT for details.
//LIC// 
#include "flxbase\scopes.h"
#include "flxbase\FlxDataSystem.h"
#include "flxbase\scripts.h"
#include "flxbase\command_manager.h"
#include "flxbase\string_utils.h"
#include "flxbase\event_hooks.h"
#include "flxbase\event_manager.h"
#include "flxbase\object_manager.h"
#include "flxbase\condition_manager.h"

FLX_LINKAGE FlxObject *flx_dependency_root;

/*****************************************************************************/

FLX_LINKAGE string flx_unique_name(void){

  return tmpnam(NULL);

} /* flx_unique_name */

/*****************************************************************************/

FLX_LINKAGE bool flx_type_convert(const string &s,FlxObject *&op){
  
  return flx_object_convert(s,op,"FlxObject");
} 

/*****************************************************************************/

void flx_add_object_source(FlxObject *dependent,FlxObject *source){
  string cur_function="flx_add_object_source";

  /* We don't want to have a source with a scope nested inside that of
     the dependent, because that creates the possibility of having the
     source be deleted while the dependent is still in use. Thus, we
     check for this and generate an error if it occurs. */
  
  if(is_nested_scope(source->scope(),dependent->scope())){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Can't make '"+dependent->name()+"' dependent on '"+source->name()+"' because the scope for '"+source->name()+"' is nested inside the scope for '"+dependent->name()+"'");
  } else {  
    flx_data->write_message(FLX_DATADDEBUG,cur_function,"Marking '"+dependent->name()+"' as dependent on '"+source->name()+"'");
    source->add_dependent(dependent);
    dependent->add_source(source);
  }

} /* flx_add_object_source */

/*****************************************************************************/
/* This function is for use with objects having an "real" update
   function (i.e., one that actually does something). There are
   basically two scenarios for updating such an object. If it
   ultimately depends on flx_dependency_root, then it will get updated
   automatically at the beginning of each trial. Otherwise, we need to
   "manually" update it each time there is some change to it. Calling
   this function with the object as the argument will take care of
   this manual updating if appropriate. */
void flx_process_dependencies(FlxObject *object){
  string cur_function="flx_process_dependencies";

  flx_data->write_message(FLX_DATADDEBUG,cur_function,"Processing dependencies for '"+object->name()+"'");
  if(object->is_static()){
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Object is static, updating now");
    object->update_object_and_dependents();
  } else {
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Object is non-static, will be updated at the beginning of the next trial");
  }

} /* flx_process_dependencies */

/*****************************************************************************/

bool do_update_object(FlxObject *object,int data_level){
  string cur_function="do_update_object";

  flx_data->write_message(data_level,cur_function,"Updating object '"+object->name()+"'");
  object->update_object_and_dependents();
  return true;

} /* do_update_object */

/*****************************************************************************/

bool update_dependencies(void){

  return do_update_object(flx_dependency_root,FLX_DATAHOOK);

} /* update_dependencies */

/*****************************************************************************/

void remove_objects_by_scope(const string &tag){
  string cur_function="remove_objects_by_scope";
  FlxObject **foh;

  flx_data->write_message(FLX_DATADDEBUG,cur_function,"Removing objects with scope '"+tag+"'");
  while((foh=flx_object_list.find_scope(tag,true))){
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Removing object '"+(*foh)->name()+"'");
    delete(*foh);
    flx_object_list.remove();
  }
  flx_object_list.rewind();

} /* remove_objects_by_scope */

/*****************************************************************************/

bool Update(FlxObject *object){ 

  return do_update_object(object,FLX_DATASCRIPT);

} /* Update */

/*****************************************************************************/

class UpdateEvent : public FlxEvent {
  FlxObject *d_object;
public:
  UpdateEvent(const string &name,FlxObject *object) : FlxEvent(name), d_object(object) {}
  ~UpdateEvent() {}
  bool update(void){ return true; }
  void execute(void){ do_update_object(d_object,FLX_DATAEVENT); }
};

/*****************************************************************************/
                                                                                
bool NewUpdateEvent(string *name,FlxObject *object){
  string cur_function="NewUpdateEvent";
  UpdateEvent *ue;

  flx_data->write_message(FLX_DATASCRIPT,cur_function,"Creating new UpdateEvent '"+*name+"'");
  ue=new UpdateEvent(*name,object);
  flx_add_object_source(ue,object);
  return true;
                                                                 
} /* NewUpdateEvent */

/*****************************************************************************/

class EqualsCondition : public FlxCondition {
  FlxBaseScalarObject *d_operand1, *d_operand2;
  bool d_triggered;
public:
  EqualsCondition(const string &name,FlxBaseScalarObject *operand1,FlxBaseScalarObject *operand2) : FlxCondition(name), d_operand1(operand1), d_operand2(operand2), d_triggered(false) {}
  ~EqualsCondition(void) {}
  bool evaluate(void) {
      flx_data->write_message(FLX_DATADDDEBUG,"EqualsCondition::evaluate","Evaluating equals condition");
    if(d_triggered){
      return false;
    } else {
      if(d_operand1->compare(d_operand2)){
	d_triggered=true;
	return true;
      } else {
	return false;
      }
    }
  }
  void reset(void) { d_triggered=false; }
};

/*****************************************************************************/

FlxCondition *equals_condition_parser(const string &keyword,int arg_count,string *arg_array){
  string cur_function="equals_condition_parser";
  int cur_arg=0;
  string clean_s;
  FlxBaseScalarObject *operand1, *operand2;
  string name;
  FlxCondition *cp;
  
  // check if we have the right number of operands
  if(arg_count!=2){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": Bad equals condition");
    return NULL;
  } 

  // parse the first operand
  if((arg_array[cur_arg])[0]=='$'){
    clean_s=(arg_array[cur_arg]).substr(1);
  } else {
    clean_s=arg_array[cur_arg];
  }
  operand1=flx_get_object_by_name<FlxBaseScalarObject>(clean_s);
  if(!operand1){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": First operand of equals condition must be a variable, but there is no variable named '"+arg_array[cur_arg]+"'");
    return NULL;
  }

  // parse the second operand
  cur_arg++;
  if(!(operand2=operand1->create_from_string(arg_array[cur_arg]))){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": Second operand of equals condition cannot be converted to the same type as the first operand");
    return NULL;
  }

  name=flx_unique_name();
  flx_data->write_message(FLX_DATADDEBUG,cur_function,"Creating new equals condition named '"+name+"'");
  cp=new EqualsCondition(name,operand1,operand2);
  flx_add_object_source(cp,operand1);
  flx_add_object_source(cp,operand2);
  return cp;
  
} /* equals_condition_parser */

/*****************************************************************************/

bool DumpObjectList(void){

  flx_object_list.dump_list();
  return true;

} /* DumpObjectList */

/*****************************************************************************/

void flx_object_manager_init(void){
  
  flx_dependency_root=new FlxObject("root");
  flx_add_command("Update",Update);
  flx_add_command("UpdateEvent",NewUpdateEvent);
  flx_add_command("DumpObjectList",DumpObjectList);
  flx_add_condition_type("equals",equals_condition_parser);
  flx_add_event_hook("update_dependencies",update_dependencies,FLX_EVENT_HOOK_PRETRIAL);
  flx_add_scope_cleanup_hook("remove_objects_by_scope",remove_objects_by_scope);

} /* flx_object_manager_init */

/*****************************************************************************/
