//LIC// FLXLab v2.5 - A program for running psychology experiments.  
//LIC// Copyright (C) 2010 Todd R. Haskell (todd.haskell@wwu.edu) 
//LIC// 
//LIC// Use and distribution is governed by the terms of the 
//LIC// GNU General Public License. Certain portions of the 
//LIC// program may be subject to other licenses as well. See 
//LIC// the file LICENSE.TXT for details.
//LIC// 
#include <fstream>
#include <sstream>
#include "flxbase\string_utils.h"
#include "flxbase\scripts.h"
#include "flxbase\FlxStack.h"
#include "flxbase\event_hooks.h"
#include "flxbase\event_manager.h"
#include "flxbase\FlxDataSystem.h"
#include "flxbase\data_manager.h"
#include "flxbase\command_manager.h"
#include "flxbase\scopes.h"
#include "flxbase\strings.h"
#include "flxbase\flxbase.h"
#include "flxbase\file_system.h"
#include "flxbase\OS_specific_file_system.h"

#define FLX_MAX_COMMAND_ARGS 10

/* these are defined in event_manager.cpp, but are not declared in the
   header file because they are not intended to be generally available */
extern bool in_session;
extern string session_scope;

ifstream *script;
FlxStack<ifstream *> script_stack;
FLX_LINKAGE string flx_cur_script_command;
int script_count, flx_cur_script_line;

/*****************************************************************************/

bool flx_get_script_line(string &s){ 
  bool return_val;

  if(return_val=(getline(*script,s)!=0)){
    s=flx_string_strip_white(s);
    flx_cur_script_line++;
  }
  return return_val;

} /* flx_get_script_line */

/*****************************************************************************/

void flx_evaluate_script_line(const string &s){
  string cur_function="flx_evaluate_script_line";
  int argc;
  string argv[FLX_MAX_COMMAND_ARGS], temp;
  char quote_char;

  if(!(s.size()==0) && !(s[0]=='#')){ // skip blank lines and comments
	istringstream line_stream(s.c_str());
	
	// parse the line into a command name and arguments
	line_stream >> flx_cur_script_command;
	argc=0;
	while(argc<FLX_MAX_COMMAND_ARGS && line_stream >> argv[argc] && !line_stream.fail()){
	  // check for quoted arguments
	  if(argv[argc][0]=='"' || argv[argc][0]=='\''){
		  quote_char=argv[argc][0];
	    // check to make sure we don't have a single quoted word
	    if(!(argv[argc].size()>1 && argv[argc][argv[argc].size()-1]==quote_char)){
	      // get the rest of the argument
	      getline(line_stream,temp,quote_char);
	      argv[argc]+=(temp+quote_char); /* need to replace the quote
					  that getline strips off */
	    }
	    argv[argc]=flx_string_strip_quotes(argv[argc]);
	  }
	  argc++;
	}
	// check if there was more to process
	if(!line_stream.fail()){
	  line_stream >> temp;
	  if(!line_stream.fail()){
	    flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": Command is too long; truncating");
	  }
	}	
	flx_execute_command(flx_cur_script_command,argc,argv);
      }

} /* flx_evaluate_script_line */

/*****************************************************************************/

FLX_LINKAGE bool flx_read_script(void){
  string cur_function="flx_read_script";
  bool return_val;

  /* If no script file is specified on the command line, it is set to
     <flx_default_script_name>; we want to treat this case differently than
     a script file that doesn't exist */
  if(flx_script_file==flx_default_script_file){
    flx_data->write_message(FLX_DATAERROR,cur_function,"No script file specified");
    return false;
  }

  flx_data->write_message(FLX_DATAINFO,cur_function,"Executing script file '"+flx_script_file+"'");  
  flx_data->new_queue("data.txt");
  return_val=flx_read_script_file(flx_script_file,false,true);
  flx_data->delete_queue();
  return return_val;

} /* flx_read_script */

/*****************************************************************************/
/* The second argument here controls whether the scope associated with
   the script is ended or deleted after the script has been
   executed. If it's true (as for config files), the scope is just
   ended. If it's false (as with script files), the scope is also
   deleted. The third argument here is optional; if <true> then the
   directory containing the script file is added to the list of search
   directories. The default is <false>. This argument provides a way
   to keep the config directory from getting added to the list of
   search directories when each config file is executed. */

bool flx_read_script_file(const string &file_name,bool config_script,bool add_search_dir){
  string cur_function="flx_read_script_file";
  string scope_name;
  string expanded_file_name, next_line;

  expanded_file_name=flx_expand_filename(file_name);
  if(expanded_file_name.empty()){
	  flx_data->write_message(FLX_DATAERROR,cur_function,"Can't find script file '"+file_name+"'");
	return false;
  } else {
    script=new ifstream(expanded_file_name.c_str());
    if(!*script){
      flx_data->write_message(FLX_DATAERROR,cur_function,"Can't open script file '"+file_name+"'"); 
      return false;
    } else {
      script_count++;
      flx_cur_script_line=1;
      begin_data_file_context();
      scope_name=file_name+"."+flx_convert_to_string(script_count);
      flx_begin_scope(scope_name);
      if(add_search_dir){
	flx_add_search_directory(flx_extract_path(expanded_file_name));
      }

      while(flx_get_script_line(next_line) && !abort_flxlab){
	flx_evaluate_script_line(next_line);
      }
      script->close();
      delete script;
      if(config_script){
	flx_end_scope();
      } else {
	flx_delete_scope(scope_name);
      }
      end_data_file_context();
      return true;
    }
  }
  
} /* flx_read_script_file */

/*****************************************************************************/

bool ReadScriptFile(string *file_name){
  string cur_function="ReadScriptFile";
  bool return_val;

  flx_data->write_message(FLX_DATASCRIPT,cur_function,"Reading script file '"+*file_name+"'");
  script_stack.push(script);
  return_val=flx_read_script_file(*file_name,false,true);
  script=script_stack.pop();
  return return_val;

} /* ReadScriptFile */

/*****************************************************************************/

bool SetScriptFile(string *file_name){
  string cur_function="SetScriptFile";
  
  flx_data->write_message(FLX_DATASCRIPT,cur_function,"Setting script file to '"+*file_name+"'");
  flx_script_file=*file_name;
  return true;

} /* SetScriptFile */

/*****************************************************************************/

void flx_scripts_init(void){

  flx_add_command("ReadScriptFile",ReadScriptFile);
  flx_add_command("SetScriptFile",SetScriptFile);
  script_count=0;

} /* flx_scripts_init */

/*****************************************************************************/
