//LIC// FLXLab v2.5 - A program for running psychology experiments.  
//LIC// Copyright (C) 2010 Todd R. Haskell (todd.haskell@wwu.edu) 
//LIC// 
//LIC// Use and distribution is governed by the terms of the 
//LIC// GNU General Public License. Certain portions of the 
//LIC// program may be subject to other licenses as well. See 
//LIC// the file LICENSE.TXT for details.
//LIC// 
#include "flxbase\string_utils.h"
#include "flxbase\FlxDataSystem.h"

/****************************************************************************/

FLX_LINKAGE int flx_string_compare_ignore_case(const string &string1,const string &string2){
  // Similar to the standard <strcmp>, except it ignores case.
 
  unsigned int i;
  
  if(string1.size()<string2.size()){
    return -1;
  } else if(string1.size()>string2.size()){
    return 1;
  } else { // strings are the same length
    for(i=0;i<string1.size();i++){
      if(toupper(string1[i])!=toupper(string2[i])){
	return toupper(string1[i])-toupper(string2[i]);
      }
    }
    // strings matched for their full length
    return 0;
  } 

} /* flx_string_compare_ignore_case */

/****************************************************************************/

FLX_LINKAGE bool flx_string_to_bool(const string &s,bool *err_flag){
  /* Converts <s> to a boolean value and returns the result. Acceptable
     formats include T, True, TRUE, and true. */
  bool dummy;
  bool *local_flag;

  if(err_flag==NULL){
    local_flag=&dummy;
  } else {
    local_flag=err_flag;
  }
  if(s=="T" || !flx_string_compare_ignore_case(s,"true")){
   *local_flag=false;
   return true;
  } else if(s=="F" || !flx_string_compare_ignore_case(s,"false")){
    *local_flag=false;
    return false;
  } else {
    *local_flag=true;
    return true;
  }

} /* flx_string_to_bool */

/****************************************************************************/

FLX_LINKAGE int flx_string_to_int(const string &s,bool *err_flag){
  /* Converts <s> to an integer and returns the result. If <err_flag>
     is provided and is non-null, it is set to TRUE for successful
     conversion, FALSE otherwise. */
  int result;
  unsigned int conv_count;
  bool dummy;
  bool *local_flag;
  
  if(err_flag==NULL)
    local_flag=&dummy;
  else
    local_flag=err_flag;
  *local_flag=!(sscanf(s.c_str(),"%d%n",&result,&conv_count)==1 && conv_count==s.size());
  if(result==INT_MAX || result==INT_MIN) *local_flag=true; // a check for overflow
  return result;

} /* flx_string_to_int */

/****************************************************************************/

FLX_LINKAGE long flx_string_to_long(const string &s,bool *err_flag){
  /* Converts <s> to a long integer and returns the result. If <err_flag>
     is provided and is non-null, it is set to TRUE for successful
     conversion, FALSE otherwise. */
  long result;
  unsigned int conv_count;
  bool dummy;
  bool *local_flag;
  
  if(err_flag==NULL)
    local_flag=&dummy;
  else
    local_flag=err_flag;
  *local_flag=!(sscanf(s.c_str(),"%ld%n",&result,&conv_count)==1 && conv_count==s.size());
  if(result==LONG_MAX || result==LONG_MIN) *local_flag=true; // a check for overflow
  return result;

} /* flx_string_to_long */

/****************************************************************************/

FLX_LINKAGE long long flx_string_to_long_long(const string &s,bool *err_flag){
  /* This isn't fully implemented right now, because I don't know any
	ANSI-compatible way to convert a string to a long long that
	actually detects errors (might be able to do this with C++
	streams; have to check that out) */

  return flx_string_to_long(s,err_flag);

} /* flx_string_to_long_long */

/****************************************************************************/

FLX_LINKAGE float flx_string_to_float(const string &s,bool *err_flag){
   /* Converts <s> to a float and returns the result. If <err_flag>
     is provided and is non-null, it is set to TRUE for successful
     conversion, FALSE otherwise. */

  float result;
  unsigned int conv_count;
  bool dummy;
  bool *local_flag;
  
  if(err_flag==NULL)
    local_flag=&dummy;
  else
    local_flag=err_flag;
  *local_flag=!(sscanf(s.c_str(),"%g%n",&result,&conv_count)==1 && conv_count==s.size());
  return result;

} /* flx_string_to_float */

/****************************************************************************/

FLX_LINKAGE double flx_string_to_double(const string &s,bool *err_flag){
   /* Converts <s> to a double and returns the result. If <err_flag>
     is provided and is non-null, it is set to TRUE for successful
     conversion, FALSE otherwise. */

  double result;
  unsigned int conv_count;
  bool dummy;
  bool *local_flag;
  
  if(err_flag==NULL)
    local_flag=&dummy;
  else
    local_flag=err_flag;
  *local_flag=!(sscanf(s.c_str(),"%lg%n",&result,&conv_count)==1 && conv_count==s.size());
  return result;

} /* flx_string_to_double */

/****************************************************************************/

FLX_LINKAGE string flx_string_strip_white(const string &s){
  /* Returns <s> with any surrounding white space removed. */
  int front_matches=0, back_matches=0;
  
  if(s.size()>0){
	while(isspace(s[front_matches])) front_matches++;
	while(isspace(s[s.size()-1-back_matches])) back_matches++;
  }

  return string(s,front_matches,s.size()-back_matches-front_matches);

} /* flx_string_strip_white */

/****************************************************************************/

FLX_LINKAGE string flx_string_strip_parens(const string &s){
  /* Returns <s> with any enclosing parentheses removed (including multiple
     pairs of enclosing parentheses, e.g. ((s)) ) */

  bool match;
  char *front, *back, *cur_pos;
  int depth=0;
  int num_matches=0;
  
  front=const_cast<char *>(s.c_str());
  back=front+s.size()-1;
  do{
    match=false;
    if(*front=='('){ // leading character is open parens
      depth++;
      // find the matching close parens (if it's present!)
      for(cur_pos=front+1;cur_pos<=back;cur_pos++){
	if(*cur_pos=='(') depth++;
	if(*cur_pos==')'){
	  depth--;
	  if(depth==0){
	    if(cur_pos==back){ /* the close parens matches the open 
					   parens at <front>, and is the last
					   character in the string */
	      match=true; /* a flag to repeat the process w/the next char */
	      front++;
	      back--;
	      num_matches++;
	    }
	    break; // we're done with this open paren
	  }
	}
      }
    }
  } while (match);

  /* return the subset of the string which excludes the parentheses */
  return string(s,num_matches,back-front+1);

} /* flx_string_strip_parens */

/****************************************************************************/

/* Note that this strips both single and double quotes */

FLX_LINKAGE string flx_string_strip_quotes(const string &s){
	string s2;

	if((s[0]=='"' && s[s.size()-1]=='"') || (s[0]=='\'' && s[s.size()-1]=='\'')){
		s2=s.substr(1,s.size()-2);
	} else {
		s2=s;
	}
	return s2;

} /* flx_string_strip_quotes */

/****************************************************************************/

