//LIC// FLXLab v2.5 - A program for running psychology experiments.  
//LIC// Copyright (C) 2010 Todd R. Haskell (todd.haskell@wwu.edu) 
//LIC// 
//LIC// Use and distribution is governed by the terms of the 
//LIC// GNU General Public License. Certain portions of the 
//LIC// program may be subject to other licenses as well. See 
//LIC// the file LICENSE.TXT for details.
//LIC// 
#include <time.h>
#include "flxbase\time.h"
#include "flxbase\strings.h"
#include "flxbase\logical_and_numeric_types.h"
#include "flxbase\command_manager.h"
#include "flxbase\scripts.h"
#include "flxbase\event_manager.h"
#include "flxbase\condition_manager.h"
#include "flxbase\scopes.h"
#include "flxbase\event_hooks.h"
#include "flxbase\object_manager.h"
#include "flxbase\FlxSetting.h"

/*****************************************************************************/

string flx_get_date(void){
  time_t timeval;
  string date_string;

  time(&timeval);
  date_string=ctime(&timeval);
  return date_string.substr(0,date_string.size()-1);

} /* flx_get_date */

/*****************************************************************************/

class DelayEvent : public FlxEvent {
  long *d_delay_time;
public:
  DelayEvent(const string &name,long *delay_time) : FlxEvent(name), d_delay_time(delay_time){}
  ~DelayEvent(void) {}
  void execute(void);
};

void DelayEvent::execute(void){
  string cur_function="DelayEvent::execute";

  flx_data->write_message(FLX_DATAEVENT,cur_function,d_name);
  this->do_generic_event_processing();

  flx_delay(*d_delay_time);
}

/*****************************************************************************/

bool NewDelayEvent(string *name,long *delay){
  string cur_function="NewDelayEvent";
  DelayEvent *dep;

  flx_data->write_message(FLX_DATASCRIPT,cur_function,"Creating new DelayEvent '"+*name+"'");
  dep=new DelayEvent(*name,delay);
  flx_add_scalar_source(dep,delay);
  return true;

} /* NewDelayEvent */

/*****************************************************************************/

class TimeCondition : public FlxCondition {
  long *d_trigger_time;
  bool d_primed, d_triggered;
public:
  TimeCondition(const string &name,long *trigger_time) : FlxCondition(name), d_trigger_time(trigger_time), d_primed(false), d_triggered(false) {}
  ~TimeCondition(void) {}
  bool evaluate(void);
  void reset(void) { d_primed=false; d_triggered=false; }
};

/* We have to do a little extra work to make conditions like 
   'whenever time 1000' behave in a sensible way. Ideally, such
   conditions would trigger each time the event time passes from
   below 1000 to above 1000, but they should not trigger
   continuously while the time is above 1000. To implement this
   behavior, we use a two-step trigger sequence; the first step
   in the sequence (priming) occurs whenever the time is below
   the trigger time. The second step (triggering) occurs when
   the condition has been primed, and now the time is above the
   trigger time. */
bool TimeCondition::evaluate(void){
  string cur_function="TimeCondition::evaluate";
 
   if(d_primed && !d_triggered && (flx_event_time>*d_trigger_time)){
    flx_data->write_message(FLX_DATADEBUG,cur_function,"Triggering");
    d_triggered=true;
    return true;
  } else {
    if(!d_primed && (flx_event_time<*d_trigger_time)){
      flx_data->write_message(FLX_DATADEBUG,cur_function,"Priming");
       d_primed=true;
     }
     return false;
  }
}

/*****************************************************************************/

FlxCondition *time_condition_parser(const string &keyword,int arg_count,string *arg_array){
  string cur_function="time_condition_parser";
  long *trigger_time=NULL;
  string name;
  FlxCondition *cp;

  // check for the right number of arguments
  if(arg_count!=1){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": Bad time condition");
      return NULL;
  } 

  // check that the argument can be converted to a number
  flx_type_convert(arg_array[0],trigger_time);
  if(!trigger_time){
      flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": Can't convert '"+arg_array[0]+"' to a number of milliseconds");
      return NULL;
  }

  // check that the number is positive
  if(*trigger_time<0){
      flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": Time value can't be negative");
      return NULL;
  }

  // create the condition
  name=flx_unique_name();
  flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Creating new time condition named '"+name+"'");
  cp=new TimeCondition(name,trigger_time);
  flx_add_scalar_source(cp,trigger_time);
  return cp;

} /* time_condition_parser */

/*****************************************************************************/

FLX_LINKAGE void flx_delay(long delay){
  string cur_function="flx_delay";
  long long start;	

  start=flx_get_time();
  while(flx_get_time()-start<delay){
    flx_execute_event_hooks(FLX_EVENT_HOOK_WITHINEVENT);
    if(flx_force_exit) return;
  }

} /* flx_delay */

/*****************************************************************************/

bool UseXSeconds(void){
  string cur_function="UseXSeconds";

  // we ignore any requests that would not change the current setting
  if(flx_cur_script_command=="UseMicroseconds"){
	  if(!flx_microseconds_enabled){
		flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": Microsecond timer is not enabled");
		return false;
	  } else {
		  flx_data->write_message(FLX_DATASCRIPT,cur_function,"Using microsecond times");
		  if(!flx_using_microseconds){
		    flx_change_global_setting(&flx_using_microseconds,true,"flx_using_microseconds");
		    flx_get_microseconds();
		  } return true;
	  }
  } else if(flx_cur_script_command=="UseMilliseconds"){
	flx_data->write_message(FLX_DATASCRIPT,cur_function,"Using millisecond times");
	if(flx_using_microseconds){
	  flx_change_global_setting(&flx_using_microseconds,false,"flx_using_microseconds");
	  flx_get_milliseconds();
	}
	return true;
  } else {
	  flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": UseXSeconds called via unknown name "+flx_cur_script_command);  
    return false;
  }

} /* UseXSeconds */

/*****************************************************************************/

void flx_time_init(void){
	string cur_function="flx_time_init";
  
  flx_add_command("DelayEvent",NewDelayEvent);
  flx_add_command("UseMilliseconds",UseXSeconds);
  flx_add_command("UseMicroseconds",UseXSeconds);
  flx_add_condition_type("time",time_condition_parser);
  	
  if(flx_microseconds_enabled){
	flx_data->write_message(FLX_DATAINFO,cur_function,"Microsecond timer enabled");
  } else {
	flx_data->write_message(FLX_DATAINFO,cur_function,"Microsecond timer not enabled");
  }

} /* flx_time_init */

/*****************************************************************************/

