//LIC// FLXLab v2.5 - A program for running psychology experiments.  
//LIC// Copyright (C) 2010 Todd R. Haskell (todd.haskell@wwu.edu) 
//LIC// 
//LIC// Use and distribution is governed by the terms of the 
//LIC// GNU General Public License. Certain portions of the 
//LIC// program may be subject to other licenses as well. See 
//LIC// the file LICENSE.TXT for details.
//LIC// 
#include <flxbase.h>
#include <portaudio.h>
#include "flxsound/sound_driver.h"
#include "flxsound/sound_buffer.h"

using namespace std;

/* this allows 99 audio streams to be open simultaneously, which
   should be more than enough */
#define MAX_STREAMS 100

class FlxStreamInfo {
public:
  PaStream *d_stream;
  FlxSoundBuffer *d_buffer;
  bool d_is_active;
  FlxStreamInfo(PaStream *stream,FlxSoundBuffer *buffer) : d_stream(stream), d_buffer(buffer), d_is_active(false) {}
  ~FlxStreamInfo(void){}
};

FlxStreamInfo *stream_info_array[MAX_STREAMS]={NULL};

FlxSoundDriver driver;
FlxSoundDriver *flx_sound_driver=&driver;

long *flx_audio_latency;

/*****************************************************************************/

int allocate_stream(PaStream *stream,FlxSoundBuffer *buffer){
  string cur_function="allocate_stream";
  int i;

  // this finds the first empty slot in the array
  for(i=1;i<MAX_STREAMS;i++){
    if(stream_info_array[i]==NULL){
      stream_info_array[i]=new FlxStreamInfo(stream,buffer);
      return i;
    }
  }

  // no available slots in the array
  flx_data->write_message(FLX_DATAERROR,cur_function,"Can't allocate an audio stream number, too many audio streams already open");
  return 0;

} /* allocate_stream */

/*****************************************************************************/

void free_stream(int stream_number){

  delete stream_info_array[stream_number];
  stream_info_array[stream_number]=NULL;

} /* free_stream */

/*****************************************************************************/

bool stream_is_valid(int stream_number){

  return (stream_number>0 && stream_number<MAX_STREAMS && stream_info_array[stream_number]!=NULL) ? true : false;

} /* stream_is_valid */

/*****************************************************************************/

#define CHECK_STREAM_NUM(failure_value) 		\
  if(stream_num<=0 || stream_num>=MAX_STREAMS){ \
  flx_data->write_message(FLX_DATAERROR,cur_function,"Bad audio stream number"); \
  return failure_value; \
  } else if(stream_info_array[stream_num]==NULL){	\
    return failure_value;				\
  }

#define GET_STREAM_INFO(info_type,failure_value)	\
  CHECK_STREAM_NUM(failure_value)			\
  return stream_info_array[stream_num]->info_type;

/*****************************************************************************/

PaStream *get_stream(int stream_num){
  string cur_function="get_stream";

  GET_STREAM_INFO(d_stream,NULL);
     
} /* get_stream */

/*****************************************************************************/

FlxSoundBuffer *get_buffer(int stream_num){
  string cur_function="get_buffer";

  GET_STREAM_INFO(d_buffer,NULL);
     
} /* get_buffer */

/*****************************************************************************/

bool get_is_active(int stream_num){
  string cur_function="get_is_active";

  GET_STREAM_INFO(d_is_active,false);
     
} /* get_is_active */

/*****************************************************************************/

void set_is_active(int stream_num,bool state){
  string cur_function="set_is_active";

  CHECK_STREAM_NUM();

  stream_info_array[stream_num]->d_is_active=state;
     
} /* set_is_active */

/*****************************************************************************/

static int paOutputCallback(const void *inputBuffer, void *outputBuffer,
		      unsigned long framesPerBuffer,
		      const PaStreamCallbackTimeInfo* timeInfo,
		      PaStreamCallbackFlags statusFlags,
		      void *userData ){
  FlxSoundBuffer *sound_buffer;
  long bytes_to_transfer;
  int retVal;

  sound_buffer=static_cast<FlxSoundBuffer *>(userData);
  bytes_to_transfer=sound_buffer->bytes_per_frame()*framesPerBuffer;

  /* If we have less than a complete buffer to transfer, pad the remaining
     space with 0 bytes. This should correspond to silence for signed samples
     at least. */
  if(bytes_to_transfer>=sound_buffer->data_bytes()-sound_buffer->cur_pos()){
    memcpy(outputBuffer,sound_buffer->data()+sound_buffer->cur_pos(),sound_buffer->data_bytes()-sound_buffer->cur_pos());
    memset(outputBuffer,0,bytes_to_transfer-(sound_buffer->data_bytes()-sound_buffer->cur_pos()));
    retVal=paComplete;
  } else {
    memcpy(outputBuffer,sound_buffer->data()+sound_buffer->cur_pos(),bytes_to_transfer);
    sound_buffer->increment_cur_pos(bytes_to_transfer);
    retVal=paContinue;
  }

  return retVal;

} /* paOutputCallback */

/*****************************************************************************/

static int paInputCallback(const void *inputBuffer, void *outputBuffer,
		      unsigned long framesPerBuffer,
		      const PaStreamCallbackTimeInfo* timeInfo,
		      PaStreamCallbackFlags statusFlags,
		      void *userData ){
  FlxSoundBuffer *sound_buffer;
  long bytes_to_transfer;
  unsigned int i;
  int j;
  int retVal;
  
  sound_buffer=static_cast<FlxSoundBuffer *>(userData);
  bytes_to_transfer=sound_buffer->bytes_per_frame()*framesPerBuffer;

  /* if we have more bytes to transfer than space in the buffer,
     just transfer what we can and stop the recording */
  if(bytes_to_transfer>=sound_buffer->data_bytes()-sound_buffer->cur_pos()){
    memcpy(sound_buffer->data()+sound_buffer->cur_pos(),inputBuffer,sound_buffer->data_bytes()-sound_buffer->cur_pos());
    sound_buffer->increment_cur_pos(sound_buffer->data_bytes()-sound_buffer->cur_pos());
    retVal=paComplete;
  } else {
    memcpy(sound_buffer->data()+sound_buffer->cur_pos(),inputBuffer,bytes_to_transfer);
    sound_buffer->increment_cur_pos(bytes_to_transfer);
    retVal=paContinue;
  }
  /* compute the average sound level for the data in inputBuffer */
  for(i=0;i<framesPerBuffer;i++){
    for(j=0;j<sound_buffer->channels();j++){
      switch(sound_buffer->bits()){
      case 8:

	break;
      case 16:

	break;
      default:
	// should never get here
	break;
      }
    }
  }
  return retVal;

} /* paInputCallback */

/*****************************************************************************/

void FlxSoundDriver::initialize(void){
  string cur_function="FlxSoundDriver::initialize";

  if(Pa_Initialize()==paNoError){
    flx_data->write_message(FLX_DATADEBUG,cur_function,"Initialized sound system");
  } else {
    flx_data->write_message(FLX_DATAERROR,cur_function,"Can't initialize sound system");
  }

} /* FlxSoundDriver::initialize */

/*****************************************************************************/

void FlxSoundDriver::terminate(void){
  string cur_function="FlxSoundDriver::terminate";

  if(Pa_Terminate()==paNoError){
    flx_data->write_message(FLX_DATADEBUG,cur_function,"Terminated sound system");
  } else {
    flx_data->write_message(FLX_DATAERROR,cur_function,"Can't terminate sound system");
  }

} /* FlxSoundDriver::terminate */

/*****************************************************************************/

int FlxSoundDriver::open_stream(FlxSoundBuffer *sound_buffer){
  string cur_function="FlxSoundDriver::open_stream";
  PaStream *stream;
  PaStreamParameters stream_params;
  int stream_num;
  PaError result;
  PaTime latency;

  stream_params.channelCount=sound_buffer->channels();  
  switch(sound_buffer->bits()){
  case 8:
    stream_params.sampleFormat=paUInt8;
    break;
  case 16:
    stream_params.sampleFormat=paInt16;
    break;
  default:
    flx_data->write_message(FLX_DATAERROR,cur_function,"Bad audio sample format");
    return 0;
    break;
  }
  stream_params.suggestedLatency=static_cast<double>(*flx_audio_latency)/1000;
  stream_params.hostApiSpecificStreamInfo=NULL;

  switch(sound_buffer->stream_type()){
  case FLX_OUTPUT_STREAM:
    stream_params.device=Pa_GetDefaultOutputDevice();
    result=Pa_OpenStream(&stream,NULL,&stream_params,sound_buffer->rate(),0,0,&paOutputCallback,sound_buffer);
    break;
  case FLX_INPUT_STREAM:
    stream_params.device=Pa_GetDefaultInputDevice();
    result=Pa_OpenStream(&stream,&stream_params,NULL,sound_buffer->rate(),0,0,&paInputCallback,sound_buffer);
    break;
  default:
    flx_data->write_message(FLX_DATAERROR,cur_function,"Bad stream type specification");
    return 0;
    break;
  }
  if(result!=paNoError){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Can't open audio stream");
    return 0;
  }

  stream_num=allocate_stream(stream,sound_buffer);

  switch(sound_buffer->stream_type()){
  case FLX_OUTPUT_STREAM:
    latency=Pa_GetStreamInfo(stream)->outputLatency;
    break;
  case FLX_INPUT_STREAM:
    latency=Pa_GetStreamInfo(stream)->inputLatency;
    break;
  default:
    break;
  }
  flx_data->write_message(FLX_DATADDEBUG,cur_function,"Opened audio stream "+flx_convert_to_string(stream_num)+" with latency "+flx_convert_to_string(latency));

  return stream_num;

} /* FlxSoundDriver::open_stream */

/*****************************************************************************/

bool FlxSoundDriver::start_stream(int stream_number){
  string cur_function="FlxSoundDriver::start_stream";
  PaStream *stream;
  
  if(!(stream=get_stream(stream_number))){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Can't get pointer to audio stream "+flx_convert_to_string(stream_number));
    return false;
  }
  
  if(Pa_StartStream(stream)==paNoError){
    flx_data->write_message(FLX_DATADDEBUG,cur_function,"Started audio stream "+flx_convert_to_string(stream_number));
    set_is_active(stream_number,true);
    return true;
  } else {
    flx_data->write_message(FLX_DATAERROR,cur_function,"Can't start audio stream "+flx_convert_to_string(stream_number));
    return false;
  }

} /* FlxSoundDriver::start_stream */

/*****************************************************************************/

bool FlxSoundDriver::stop_stream(int stream_number){
  string cur_function="FlxSoundDriver::stop_stream";
  PaStream *stream;
  FlxSoundBuffer *buffer;

  if(!(stream=get_stream(stream_number))){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Can't get pointer to audio stream "+flx_convert_to_string(stream_number));
    return false;
  }

  if(Pa_StopStream(stream)!=paNoError){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Can't stop audio stream "+flx_convert_to_string(stream_number));
    return false;
  }
  
  // reset the buffer to the beginning
  buffer=get_buffer(stream_number);
  buffer->reset_cur_pos();

  // mark the stream as not active
  set_is_active(stream_number,false);

  flx_data->write_message(FLX_DATADDEBUG,cur_function,"Stopped audio stream "+flx_convert_to_string(stream_number));

  return true;

} /* FlxSoundDriver::stop_stream */

/*****************************************************************************/

bool FlxSoundDriver::close_stream(int stream_number){
  string cur_function="FlxSoundDriver::close_stream";
  PaStream *stream;
  bool result;

  if(!(stream=get_stream(stream_number))){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Can't get pointer to audio stream "+flx_convert_to_string(stream_number));
    return false;
  }

  result=true;

  if(Pa_CloseStream(stream)!=paNoError){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Can't close audio stream "+flx_convert_to_string(stream_number));
    result=false;
  } 

  free_stream(stream_number);

  flx_data->write_message(FLX_DATADDEBUG,cur_function,"Closed audio stream "+flx_convert_to_string(stream_number));

  return result;

} /* FlxSoundDriver::close_stream */

/*****************************************************************************/

bool FlxSoundDriver::is_valid_stream(int stream_number){

  return stream_is_valid(stream_number);

} /* FlxSoundDriver::is_valid_stream */

/*****************************************************************************/

bool FlxSoundDriver::is_active_stream(int stream_number){

  return get_is_active(stream_number);

} /* FlxSoundDriver::is_active_stream */

/*****************************************************************************/

bool stop_audio_streams(void){
  string cur_function="close_audio_streams";
  int i;
  PaStream *stream;
  bool success;

  flx_data->write_message(FLX_DATAHOOK,cur_function,"Stopping audio streams");
  success=true;
  for(i=1;i<MAX_STREAMS;i++){
    if(get_is_active(i)){
      if((stream=get_stream(i))){
	if(!flx_sound_driver->stop_stream(i)){
	  success=false;
	}
      }
    }
  }

  return success;

} /* stop_audio_streams */

/*****************************************************************************/

bool SetAudioLatency(long *latency){
  string cur_function="SetAudioLatency";

  flx_data->write_message(FLX_DATASCRIPT,cur_function,"Setting audio latency");
  flx_change_global_setting(&flx_audio_latency,latency,"flx_audio_latency");

  return true;

} /* SetAudioLatency */

/*****************************************************************************/

void flx_sound_driver_init(void){
  string cur_function="flx_play_sound_init";

  flx_add_command("AudioLatency",SetAudioLatency);
  flx_add_event_hook("stop_audio_streams",stop_audio_streams,FLX_EVENT_HOOK_POSTTRIAL);

  // this seems to be a sensible default latency to avoid output underruns
  flx_audio_latency=flx_long(50);
  
} /* flx_sound_driver_init */

/*****************************************************************************/
