//LIC// FLXLab v2.5 - A program for running psychology experiments.  
//LIC// Copyright (C) 2010 Todd R. Haskell (todd.haskell@wwu.edu) 
//LIC// 
//LIC// Use and distribution is governed by the terms of the 
//LIC// GNU General Public License. Certain portions of the 
//LIC// program may be subject to other licenses as well. See 
//LIC// the file LICENSE.TXT for details.
//LIC// 

#include <math.h>
#include <alsa\asoundlib.h>
#include <flxbase.h>
#include "flxsounddriver\sound_driver.h"

// maximum number of sounds we can have playing/recording at one time
#define FLX_MAX_PLAYBACK_BUFFERS 5
#define FLX_MAX_CAPTURE_BUFFERS 1

char *cur_playback_buffer;
long cur_bytes_remaining;

/*****************************************************************************/

typedef snd_pcm_sframes_t (*TransferOperation)(snd_pcm_t *,void *buffer,snd_pcm_uframes_t);

/*****************************************************************************/

class BufferInfo {
public:
  snd_pcm_t *d_device_handle;
  char *d_buffer;
  unsigned long d_buffer_size;
  unsigned long d_buffer_position;
  snd_pcm_uframes_t d_period_size;
  unsigned int d_bytes_per_frame;
  BufferInfo(void) : d_device_handle(NULL), d_buffer(NULL), d_buffer_size(0), d_buffer_position(0), d_period_size(0), d_bytes_per_frame(0) {}
  ~BufferInfo(void){}
};
                                                                             
/*****************************************************************************/

class AlsaDevice {
 protected:
  string d_device_name;
  snd_pcm_stream_t d_stream;
  TransferOperation d_transfer_operation;
  int d_max_buffers;
  BufferInfo *d_buffer_data;
  int d_num_buffers;
public:
  AlsaDevice(){}
  ~AlsaDevice(){}
  int prepare(void);
  bool start(int device_id,int channels,int bits,long rate,char *buffer,long buffer_size);
  int update(int);
  void stop(int); 
};

class AlsaPlaybackDevice : public AlsaDevice {
public:
  /* we need the cast here because the argument types for the reading
     and writing operations are not exactly the same (for writing the
     buffer argument is <const void *> rather than just <void *>) */
  AlsaPlaybackDevice(void){
    d_device_name="default";
    d_stream=SND_PCM_STREAM_PLAYBACK;
    d_transfer_operation=(TransferOperation) snd_pcm_writei;
    d_max_buffers=FLX_MAX_PLAYBACK_BUFFERS;
    d_buffer_data=new BufferInfo[FLX_MAX_PLAYBACK_BUFFERS];
  }
  ~AlsaPlaybackDevice(){ delete [] d_buffer_data; }
};

class AlsaCaptureDevice : public AlsaDevice {
public:
  AlsaCaptureDevice(void){
    d_device_name="default";
    d_stream=SND_PCM_STREAM_CAPTURE;
    d_transfer_operation=snd_pcm_readi;
    d_max_buffers=FLX_MAX_CAPTURE_BUFFERS;
    d_buffer_data=new BufferInfo[FLX_MAX_CAPTURE_BUFFERS];
  }
  ~AlsaCaptureDevice(){ delete [] d_buffer_data; }
};

/*****************************************************************************/

class AlsaDriver : public FlxSoundDriver {
  AlsaPlaybackDevice d_playback_device;
  AlsaCaptureDevice d_capture_device;
 public:
  AlsaDriver(void){}
  ~AlsaDriver(void){}
  bool init(void);
  void exit(void);
  int prepare_playback(void){ return d_playback_device.prepare(); }
  bool start_playback(int device_id,int channels,int bits,long rate,char *buffer,long buffer_size){ return d_playback_device.start(device_id,channels,bits,rate,buffer,buffer_size); }
  int update_playback(int device_id){ return d_playback_device.update(device_id); }
  void stop_playback(int device_id){ d_playback_device.stop(device_id); }
  int prepare_capture(void){ return d_capture_device.prepare(); }
  bool start_capture(int device_id,int channels,int bits,long rate,char *buffer,long buffer_size){ return d_capture_device.start(device_id,channels,bits,rate,buffer,buffer_size); }
  int update_capture(int device_id){ return d_capture_device.update(device_id); }
  void stop_capture(int device_id){ d_capture_device.stop(device_id); }
};

/*****************************************************************************/

AlsaDriver alsa_driver;
FlxSoundDriver *flx_sound_driver=&alsa_driver;

/*****************************************************************************/

int AlsaDevice::prepare(void){
  string cur_function="AlsaDevice::prepare";
  
  if(d_num_buffers==d_max_buffers){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Too many ALSA buffers, can't create a new one");
    return 0;
  }  
  d_buffer_data[d_num_buffers]=BufferInfo();
  d_num_buffers++;
  flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Prepared ALSA buffer "+flx_convert_to_string(d_num_buffers));
  return d_num_buffers;

} /* AlsaDevice::prepare */

/*****************************************************************************/

bool AlsaDevice::start(int device_id,int channels,int bits,long rate,char *buffer,long buffer_size){
  string cur_function="AlsaDevice::start";
  BufferInfo *cur_buffer;
  bool return_val;
  snd_pcm_hw_params_t *hw_params;
  _snd_pcm_format format;

  switch(bits){
  case 8:
    format=SND_PCM_FORMAT_U8;
    break;
  case 16:
    format=SND_PCM_FORMAT_S16_LE;
    break;
  case 24:
    format=SND_PCM_FORMAT_S24_LE;
    break;
  default:
    format=SND_PCM_FORMAT_UNKNOWN;
    break;
  }

  if(device_id>d_max_buffers){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Index to ALSA device buffer out of range (index is "+flx_convert_to_string(device_id)+", maximum is "+flx_convert_to_string(d_max_buffers)+")");
    return false;
  } else {
    cur_buffer=&(d_buffer_data[device_id-1]);
  }
  if(snd_pcm_open(&(cur_buffer->d_device_handle),d_device_name.c_str(),d_stream,SND_PCM_NONBLOCK) < 0){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Couldn't open ALSA device");
    return false;
  }


  /* 24 bit samples require 32 bits (4 bytes) of storage */
  cur_buffer->d_bytes_per_frame=(bits==24 ? channels*4 : channels*bits/8);

/* we make the period size the largest power of 2 that will still
     give a period less than 1 millisecond */
  cur_buffer->d_period_size=static_cast<snd_pcm_uframes_t>(log(static_cast<float>(rate)/1000)/log(2.0));

  snd_pcm_hw_params_malloc(&hw_params);

  // IS THE DEVICE OPEN?
  if(!cur_buffer->d_device_handle){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Can't start PCM audio device, device not open");
    return_val=false;
    // CAN WE CONFIGURE IT?
  } else if (snd_pcm_hw_params_any(cur_buffer->d_device_handle, hw_params) < 0) {
    flx_data->write_message(FLX_DATAERROR,cur_function,"Cannot configure this audio device");
    return_val=false;
    // SET THE ACCESS TYPE TO INTERLEAVED
  } else if (snd_pcm_hw_params_set_access(cur_buffer->d_device_handle, hw_params, SND_PCM_ACCESS_RW_INTERLEAVED) < 0) {
    flx_data->write_message(FLX_DATAERROR,cur_function,"Can't set access type of audio device to interleaved");
    return_val=false;
    // SET SAMPLE FORMAT
  } else if (snd_pcm_hw_params_set_format(cur_buffer->d_device_handle, hw_params, format) < 0) {
    flx_data->write_message(FLX_DATAERROR,cur_function,"Audio device can't handle "+flx_convert_to_string(bits)+" bits");
    return_val=false;
    // SET NUMBER OF CHANNELS
  } else if (snd_pcm_hw_params_set_channels(cur_buffer->d_device_handle, hw_params, channels) < 0) {
    flx_data->write_message(FLX_DATAERROR,cur_function,"Audio device can't handle "+flx_convert_to_string(channels)+" channels");
    return_val=false;
    // SET SAMPLE RATE
  } else if (snd_pcm_hw_params_set_rate(cur_buffer->d_device_handle, hw_params, rate,0) < 0) {
    flx_data->write_message(FLX_DATAERROR,cur_function,"Audio device can't handle "+flx_convert_to_string(channels)+" Hz sample rate");
    return_val=false;
    // SET THE PERIOD (FRAGMENT) SIZE IN FRAMES
  } else if(snd_pcm_hw_params_set_period_size_near(cur_buffer->d_device_handle,hw_params,&(cur_buffer->d_period_size),0)<0){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Can't set period size for audio device to "+flx_convert_to_string(cur_buffer->d_period_size)+" frames");
    return_val=false;
    // APPLY THESE PARAMETERS
  } else if (snd_pcm_hw_params(cur_buffer->d_device_handle, hw_params) < 0) {
    flx_data->write_message(FLX_DATAERROR,cur_function,"Can't apply parameters to audio device");
    return_val=false;
  } else if(d_stream==SND_PCM_STREAM_CAPTURE && snd_pcm_start(cur_buffer->d_device_handle)<0){
      flx_data->write_message(FLX_DATAERROR,cur_function,"Can't start audio device for recording");
      return_val=false;
  } else {
    cur_buffer->d_buffer=buffer;
    cur_buffer->d_buffer_size=buffer_size;
    cur_buffer->d_buffer_position=0;
    
    flx_data->write_message(FLX_DATADEBUG,cur_function,"Started ALSA buffer "+flx_convert_to_string(device_id)+" with "+flx_convert_to_string(bits)+" bit samples, "+flx_convert_to_string(channels)+" channels at a rate of "+flx_convert_to_string(rate)+" Hz, with a period size of "+flx_convert_to_string(cur_buffer->d_period_size)+" frames, and a frame size of "+flx_convert_to_string(cur_buffer->d_bytes_per_frame)+" bytes");
    return_val=true;
  }
  snd_pcm_hw_params_free(hw_params);

  return return_val;

} /* AlsaDevice::start */

/*****************************************************************************/

int AlsaDevice::update(int device_id){
  string cur_function="AlsaDevice::update";
  BufferInfo *cur_buffer;
  int result;
  snd_pcm_status_t *device_status;
  snd_pcm_state_t device_state;
  unsigned long frames_left_in_buffer, frames_free_in_device, frames_to_transfer;
  snd_pcm_sframes_t frames_used_in_device;
  long frames_transferred;

  if(device_id>d_max_buffers){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Index to ALSA device buffer out of range (index is "+flx_convert_to_string(device_id)+", maximum is "+flx_convert_to_string(d_max_buffers)+")");
      return FLX_SOUND_DRIVER_ERROR;
  } else {
    cur_buffer=&(d_buffer_data[device_id-1]);
  }

  /* find out how many frames are left in the buffer to play/record */
  frames_left_in_buffer=(cur_buffer->d_buffer_size-cur_buffer->d_buffer_position)/cur_buffer->d_bytes_per_frame;

  /* find out device status and how many frames are free and used in the
     device memory */
  snd_pcm_status_malloc(&device_status);
  result=snd_pcm_status(cur_buffer->d_device_handle,device_status);
  if(result>=0){
    device_state=snd_pcm_status_get_state(device_status);
    frames_free_in_device=snd_pcm_status_get_avail(device_status);
  }
  snd_pcm_status_free(device_status);
  if(d_stream==SND_PCM_STREAM_CAPTURE){
    result+=snd_pcm_delay(cur_buffer->d_device_handle,&frames_used_in_device);
  } else {
    frames_used_in_device=0;
  }
  if(result<0){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Error while determining device status");
    return FLX_SOUND_DRIVER_ERROR;
  }

  if(result<0){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Error while determining status of sound device");
    return FLX_SOUND_DRIVER_ERROR;
  }

  if(frames_left_in_buffer==0){
    /* For capture, this means we've run out of space to hold the incoming
       sound data, so it should be interpreted as an error condition. */
    if(d_stream==SND_PCM_STREAM_CAPTURE){
      return FLX_SOUND_DRIVER_ERROR;
    } else if(d_stream==SND_PCM_STREAM_PLAYBACK){
      /* For playback, this means all the audio data has been transferred to
       the device, but the device may still have to finish playing that data.
       Thus, we check if the device is still running. */
      if(device_state==SND_PCM_STATE_RUNNING){
	return FLX_SOUND_DRIVER_RUNNING; 
      } else {
	return FLX_SOUND_DRIVER_STOPPED;
      }
    }
  }
  
  if(d_stream==SND_PCM_STREAM_PLAYBACK){
    frames_to_transfer=(frames_free_in_device>=(cur_buffer->d_buffer_size-cur_buffer->d_buffer_position)/cur_buffer->d_bytes_per_frame ? (cur_buffer->d_buffer_size-cur_buffer->d_buffer_position)/cur_buffer->d_bytes_per_frame : frames_free_in_device);
  } else { // capture
    frames_to_transfer=(frames_used_in_device>static_cast<long>(cur_buffer->d_period_size) ? frames_used_in_device : 0);
  }
  /* use integer division to get a number of frames that is a
     multiple of the period size, unless we have less than one
     period left, in which case we transfer the total number of frames */
  if(frames_to_transfer>cur_buffer->d_period_size) frames_to_transfer=int(frames_to_transfer/cur_buffer->d_period_size)*cur_buffer->d_period_size;
  /* don't try to transfer more frames than we have left in our buffer */
  if(frames_to_transfer>frames_left_in_buffer) frames_to_transfer=frames_left_in_buffer;

  /* check if we actually need to transfer anything */
  if(frames_to_transfer==0) return FLX_SOUND_DRIVER_RUNNING;

  /* do the transfer */
  if((frames_transferred=d_transfer_operation(cur_buffer->d_device_handle,cur_buffer->d_buffer+cur_buffer->d_buffer_position,frames_to_transfer))>=0){
    cur_buffer->d_buffer_position+=frames_transferred*cur_buffer->d_bytes_per_frame;
    return frames_transferred*cur_buffer->d_bytes_per_frame;
  } else {
    flx_data->write_message(FLX_DATAERROR,cur_function,(frames_transferred==-1*EPIPE ? string("Underrun") : string("Device")) + " error while transferring sound data");
    return FLX_SOUND_DRIVER_ERROR;
  } 

} /* AlsaDevice::update */

/*****************************************************************************/

void AlsaDevice::stop(int device_id){
  string cur_function="AlsaDevice::stop";
  BufferInfo *cur_buffer;
  int i;

  if(device_id>d_max_buffers){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Index to ALSA device buffer out of range (index is "+flx_convert_to_string(device_id)+", maximum is "+flx_convert_to_string(d_max_buffers)+")");
      return;
  } else {
    cur_buffer=&(d_buffer_data[device_id-1]);
  }

  /* It is possible that the device might be stopped after only
     being prepared, without actually being started, in which
     case it won't have been configured, and we don't want to
     call snd_pcm_drop. We can check this by looking at the
     value of d_buffer, which is NULL after being prepared, and non-NULL
     once started. */
  if(cur_buffer->d_buffer) snd_pcm_drop(cur_buffer->d_device_handle);  
  snd_pcm_close(cur_buffer->d_device_handle);
  cur_buffer->d_buffer=NULL;

  /* This reclaims any contiguous space at the end of the array. At the end of
     the trial, this will have the effect of setting d_num_playback_buffers
     back to 0 */
  i=d_max_buffers-1;
  while(i>=0 && d_buffer_data[i].d_buffer==NULL) i--;
  if(i<d_num_buffers-1) d_num_buffers=i+1;

  flx_data->write_message(FLX_DATADEBUG,"AlsaDevice::stop","Stopped ALSA buffer "+flx_convert_to_string(device_id)); 

} /* AlsaDevice::stop */

/*****************************************************************************/

bool AlsaDriver::init(void){
  string cur_function="AlsaDriver::init";

  /* This function doesn't do anything; we only initialize the devices
     when they are actually needed */
  flx_data->write_message(FLX_DATADEBUG,cur_function,"Initialized ALSA driver");
  return true;

} /* AlsaDriver::init */

/*****************************************************************************/

void AlsaDriver::exit(void){
  string cur_function="AlsaDriver::exit";

  flx_data->write_message(FLX_DATADEBUG,cur_function,"Stopped ALSA driver");

} /* AlsaDriver::exit */

/*****************************************************************************/

