//LIC// FLXLab v2.5 - A program for running psychology experiments.  
//LIC// Copyright (C) 2010 Todd R. Haskell (todd.haskell@wwu.edu) 
//LIC// 
//LIC// Use and distribution is governed by the terms of the 
//LIC// GNU General Public License. Certain portions of the 
//LIC// program may be subject to other licenses as well. See 
//LIC// the file LICENSE.TXT for details.
//LIC// 

#include <flxbase.h>

#include "flxsounddriver\sound_driver.h"

#include <Dsound.h>

#define FLX_PLAYBACK_BUFFER 0
#define FLX_CAPTURE_BUFFER 1

// internal capture buffer holds 1 sec of sound
#define FLX_CAPTURE_BUFFER_LENGTH 1000
// try to transfer data every millisecond
#define FLX_CAPTURE_TRANSFER_LENGTH 1
// maximum number of sounds we can have playing at one time
#define FLX_MAX_PLAYBACK_BUFFERS 10

/*****************************************************************************/

class BufferInfo {
public:
	LPDIRECTSOUNDBUFFER d_buffer;
	DWORD d_last_play_position;
	BufferInfo(void) : d_buffer(NULL), d_last_play_position(0) {}
	BufferInfo(LPDIRECTSOUNDBUFFER buffer) : d_buffer(buffer), d_last_play_position(0) {}
	~BufferInfo(void){}
};

class DirectSoundPlaybackDevice {
protected:
	LPDIRECTSOUND d_device;
	BufferInfo d_device_buffers[FLX_MAX_PLAYBACK_BUFFERS];
	int d_num_device_buffers;
	bool d_get_device_buffer(int,LPDIRECTSOUNDBUFFER **,DWORD **);
public:
	DirectSoundPlaybackDevice() : d_num_device_buffers(0) {}
  ~DirectSoundPlaybackDevice(){}
  bool init(void);
  int prepare(void);
  bool start(int,int,int,long,char *,long);
  bool update(int);
  void stop(int);
  void exit(void){ d_device->Release(); }
};

class DirectSoundCaptureDevice {
protected:
	LPDIRECTSOUNDCAPTURE d_device;
	LPDIRECTSOUNDCAPTUREBUFFER d_device_buffer;
	DWORD d_device_buffer_size;
	unsigned long d_last_read_position;
	unsigned long d_min_transfer_size;
	char *d_program_buffer_current_position;
	long d_program_buffer_bytes_remaining;
public:
	DirectSoundCaptureDevice() : d_device_buffer(NULL) {}
  virtual ~DirectSoundCaptureDevice(){}
  bool init(void);
  bool start(int,int,long,char *,long);
  int update(void);
  void stop(void);
  void exit(void){ d_device->Release(); }
};

/*****************************************************************************/

class DirectSoundDriver : public FlxSoundDriver {
	DirectSoundPlaybackDevice d_playback_device;
	DirectSoundCaptureDevice d_capture_device;
 public:
  DirectSoundDriver(void){}
 ~DirectSoundDriver(void){}
  bool init(void);
  void exit(void);
  int prepare_playback(void){ return d_playback_device.prepare(); }
  bool start_playback(int device_buffer_descriptor,int bits,int channels,long rate,char* buffer,long buffer_size){ return d_playback_device.start(device_buffer_descriptor,bits,channels,rate,buffer,buffer_size); }
  int update_playback(int device_buffer_descriptor){ return d_playback_device.update(device_buffer_descriptor); }
  void stop_playback(int device_buffer_descriptor){ d_playback_device.stop(device_buffer_descriptor); }
  int prepare_capture(void){ return 1; } // can only have one capture buffer at a time
  bool start_capture(int device_buffer_descriptor,int bits,int channels,long rate,char *buffer,long bytes_remaining){ if(device_buffer_descriptor==1) return d_capture_device.start(bits,channels,rate,buffer,bytes_remaining); else return false; }
  int update_capture(int device_buffer_descriptor){ if(device_buffer_descriptor==1) return d_capture_device.update(); else return 0; }
  void stop_capture(int device_buffer_descriptor){ if(device_buffer_descriptor==1) d_capture_device.stop(); }
};

/*****************************************************************************/

DirectSoundDriver direct_sound_driver;
__declspec(dllexport) FlxSoundDriver *flx_sound_driver=&direct_sound_driver;

/*****************************************************************************/

bool DirectSoundPlaybackDevice::d_get_device_buffer(int device_buffer_descriptor,LPDIRECTSOUNDBUFFER **device_buffer,DWORD **last_play_position){
	string cur_function="DirectSoundPlaybackDevice::d_get_device_buffer";

	if(device_buffer_descriptor>FLX_MAX_PLAYBACK_BUFFERS){
		flx_data->write_message(FLX_DATAERROR,cur_function,"Index to DirectSound playback buffer out of range (index is "+flx_convert_to_string(device_buffer_descriptor)+", maximum is "+flx_convert_to_string(FLX_MAX_PLAYBACK_BUFFERS)+")");
		return false;
	}

	if(d_device_buffers[device_buffer_descriptor-1].d_buffer){
		if(device_buffer) (*device_buffer)=&(d_device_buffers[device_buffer_descriptor-1].d_buffer);
		if(last_play_position) (*last_play_position)=&(d_device_buffers[device_buffer_descriptor-1].d_last_play_position);
		return true;
	} else { // this should never happen, but just in case
		flx_data->write_message(FLX_DATAERROR,cur_function,"Error accessing DirectSound playback buffer "+flx_convert_to_string(device_buffer_descriptor));
		return false;
	}

} /* DirectSoundPlaybackDevice::d_get_device_buffer */

/*****************************************************************************/

bool DirectSoundPlaybackDevice::init(void){
  string cur_function="DirectSoundPlaybackDevice::init";
	HWND hwnd;

  if(DirectSoundCreate(NULL,&d_device,NULL)!=DS_OK){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Couldn't open DirectSound device for playback");
    return false;
  }

  hwnd=GetForegroundWindow();
  if(d_device->SetCooperativeLevel(hwnd, DSSCL_PRIORITY)!=NOERROR){
	flx_data->write_message(FLX_DATAERROR,cur_function,"Can't set cooperative level for DirectSound device for playback");
    return false;
  }
  return true;

} /* DirectSoundPlaybackDevice::init */

/*****************************************************************************/

bool DirectSoundCaptureDevice::init(void){
  string cur_function="DirectSoundCaptureDevice::init";

  if(DirectSoundCaptureCreate(NULL,&d_device,NULL)!=DS_OK){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Couldn't open DirectSound device for capture");
    return false;
  }
  return true;

} /* DirectSoundCaptureDevice::init */

/*****************************************************************************/

void *create_buffer_desc(int channels,int bits,long rate,long size,int buffer_type){
	static WAVEFORMATEX wfx;
	static DSBUFFERDESC dsbdesc;
	static DSCBUFFERDESC dscbdesc;

  // Set up wave format structure. 
  memset(&wfx, 0, sizeof(WAVEFORMATEX)); 
  wfx.wFormatTag = WAVE_FORMAT_PCM; 
  wfx.nChannels = channels; 
  wfx.nSamplesPerSec = rate; 
  wfx.nBlockAlign = bits*channels/8;
  wfx.nAvgBytesPerSec = wfx.nSamplesPerSec*wfx.nBlockAlign;  
  wfx.wBitsPerSample = bits; 
 
	// Set up buffer description structure
  if(buffer_type==FLX_PLAYBACK_BUFFER){
	memset(&dsbdesc, 0, sizeof(DSBUFFERDESC)); 
	dsbdesc.dwSize = sizeof(DSBUFFERDESC);
	dsbdesc.dwBufferBytes = size;
	dsbdesc.lpwfxFormat = &wfx;
	return &dsbdesc;
  } else {
	memset(&dscbdesc,0,sizeof(DSCBUFFERDESC));
	dscbdesc.dwSize = sizeof(DSCBUFFERDESC);
	dscbdesc.dwBufferBytes = size;
	dscbdesc.lpwfxFormat = &wfx;
	return &dscbdesc;
  }

} /* create_buffer_desc */

/*****************************************************************************/

int DirectSoundPlaybackDevice::prepare(void){
	string cur_function="DirectSoundPlaybackDevice::prepare";

	if(d_num_device_buffers==FLX_MAX_PLAYBACK_BUFFERS){
		flx_data->write_message(FLX_DATAERROR,cur_function,"Too many playback buffers, can't create a new one");
		return 0;
	} else {
		d_num_device_buffers++;
		flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Prepared DirectSound playback buffer "+flx_convert_to_string(d_num_device_buffers));
		return d_num_device_buffers;
	}

} /* DirectSoundPlaybackDevice::prepare */

/*****************************************************************************/

bool DirectSoundPlaybackDevice::start(int device_buffer_descriptor,int channels,int bits,long rate,char *program_buffer,long program_buffer_size){
	string cur_function="DirectSoundPlaybackDevice::start";
	VOID *block1_address=NULL, *block2_address=NULL;
	DWORD block1_size, block2_size;
	LPDIRECTSOUNDBUFFER device_buffer; 

	if(device_buffer_descriptor>d_num_device_buffers || device_buffer_descriptor<1){
		flx_data->write_message(FLX_DATAERROR,cur_function,"Request to start non-existent playback buffer");
		return false;
	}

	// Create buffer.  
	if(!SUCCEEDED(d_device->CreateSoundBuffer(static_cast<DSBUFFERDESC *>(create_buffer_desc(channels,bits,rate,program_buffer_size,FLX_PLAYBACK_BUFFER)), &device_buffer, NULL))){ 
		flx_data->write_message(FLX_DATAERROR,cur_function,"Can't create DirectSound buffer");
		return false;
	}

	// Lock the buffer.	
	if (!SUCCEEDED(device_buffer->Lock(0,program_buffer_size,&block1_address,&block1_size,&block2_address,&block2_size,0))){
		flx_data->write_message(FLX_DATAERROR,cur_function,"Error while locking DirectSound buffer");
		return false;
	}
	// Transfer the sound data
	CopyMemory(block1_address, program_buffer, block1_size); 

	// Release the data back to DirectSound.  
	if (!SUCCEEDED(device_buffer->Unlock(block1_address,block1_size,block2_address,block2_size))){
		flx_data->write_message(FLX_DATAERROR,cur_function,"Error while unlocking DirectSound buffer");
		return false;
	}

	/* Store information about the playback buffer so that the update & stop functions
		can get at it. Note that we index the array from 0, but the descriptors 
		themselves start at 1 */
	d_device_buffers[device_buffer_descriptor-1]=BufferInfo(device_buffer);

	// Start playing the sound
	device_buffer->Play(0,0,0);
	flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Starting DirectSound playback buffer "+flx_convert_to_string(d_num_device_buffers)+" to play "+flx_convert_to_string(program_buffer_size)+" bytes with "+flx_convert_to_string(bits)+" bit samples, "+flx_convert_to_string(channels)+" channels at a rate of "+flx_convert_to_string(rate)+" Hz");
	return true;

} /* DirectSoundPlaybackDevice::start */

/*****************************************************************************/

bool DirectSoundCaptureDevice::start(int channels,int bits,long rate,char *buffer,long bytes_remaining){
	string cur_function="DirectSoundCaptureDevice::start";	
	long bytes_per_ms;

	bytes_per_ms=rate*bits*channels/8000;
	d_device_buffer_size=FLX_CAPTURE_BUFFER_LENGTH*bytes_per_ms; 
	d_min_transfer_size=FLX_CAPTURE_TRANSFER_LENGTH*bytes_per_ms;

	// Create & start the buffer 
	if(!SUCCEEDED(d_device->CreateCaptureBuffer(static_cast<DSCBUFFERDESC *>(create_buffer_desc(channels,bits,rate,d_device_buffer_size,FLX_CAPTURE_BUFFER)), &d_device_buffer, NULL))){
		flx_data->write_message(FLX_DATAERROR,cur_function,"Can't create DirectSound buffer");
		d_device_buffer = NULL;
		return false;
	} else {
		d_device_buffer->Start(DSCBSTART_LOOPING);
		d_last_read_position=0;
		d_program_buffer_current_position=buffer;
		d_program_buffer_bytes_remaining=bytes_remaining;
		flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Started DirectSound capture buffer of size "+flx_convert_to_string(d_device_buffer_size)+" bytes with "+flx_convert_to_string(bits)+" bit samples, "+flx_convert_to_string(channels)+" channels at a rate of "+flx_convert_to_string(rate)+" Hz");
		return true;
	}

} /* DirectSoundCaptureDevice::start */

/*****************************************************************************/

bool DirectSoundPlaybackDevice::update(int device_buffer_descriptor){ 
	string cur_function="DirectSoundPlaybackDevice::update"; 
	LPDIRECTSOUNDBUFFER *device_buffer;
	DWORD *last_play_position, cur_play_position;
 
	if(!d_get_device_buffer(device_buffer_descriptor,&device_buffer,&last_play_position)){
		return false;
	}

	if((*device_buffer)->GetCurrentPosition(&cur_play_position,NULL)!=DS_OK){
		flx_data->write_message(FLX_DATAERROR,cur_function,"Error determining current position in DirectSound playback buffer "+flx_convert_to_string(device_buffer_descriptor));
		return false;
	}

	/* this happens when the entire buffer has been played, and the position is reset
	to the beginning of the buffer */
	if(cur_play_position<*last_play_position){
		return false;
	} else {
		*last_play_position=cur_play_position;
		return true;
	}

} /* DirectSoundPlaybackDevice::update */

/*****************************************************************************/

int DirectSoundCaptureDevice::update(void){
  string cur_function="DirectSoundCaptureDevice::update";
  DWORD cur_read_position, transfer_size;
  VOID *block1_address=NULL, *block2_address=NULL;
  DWORD block1_size, block2_size;

	if(!d_device_buffer) return 0;

 	/**** check if we're ready to transfer ****/
	// get the read position	
	if(d_device_buffer->GetCurrentPosition(NULL,&cur_read_position)!=DS_OK){
		return 0;
	}
	/* do we have enough (or too much) data to transfer? (taking into account the 
		possibility that the data has wrapped around to the beginning of the buffer) */
	transfer_size=cur_read_position-d_last_read_position+(cur_read_position<d_last_read_position ? d_device_buffer_size : 0);
	if(transfer_size<d_min_transfer_size || transfer_size>static_cast<unsigned long>(d_program_buffer_bytes_remaining)) return 0;
	// we have an appropriate amount of data, lock the buffer.
	if (!SUCCEEDED(d_device_buffer->Lock(d_last_read_position,transfer_size,&block1_address,&block1_size,&block2_address,&block2_size,0))){
		flx_data->write_message(FLX_DATAERROR,cur_function,"Error while locking DirectSound buffer");
		return 0;
	}
	// transfer the sound data
	CopyMemory(d_program_buffer_current_position,block1_address,block1_size);
	CopyMemory(d_program_buffer_current_position+block1_size,block2_address,block2_size);
	// release the data back to DirectSound. 
	if (!SUCCEEDED(d_device_buffer->Unlock(block1_address,block1_size,block2_address,block2_size))){
		flx_data->write_message(FLX_DATAERROR,cur_function,"Error while unlocking DirectSound buffer");
		return 0;
	}
	// successfully transferred sound data
	d_last_read_position+=transfer_size;
	if(d_last_read_position>=d_device_buffer_size) d_last_read_position-=d_device_buffer_size;
	d_program_buffer_current_position+=transfer_size;
	d_program_buffer_bytes_remaining-=transfer_size;
	return transfer_size;

} /* DirectSoundCaptureDevice::update */

/*****************************************************************************/

void DirectSoundPlaybackDevice::stop(int device_buffer_descriptor){
	string cur_function="DirectSoundPlaybackDevice::stop";
	LPDIRECTSOUNDBUFFER *device_buffer;
	int i;

	if(!d_get_device_buffer(device_buffer_descriptor,&device_buffer,NULL)){
		return;
	}

	(*device_buffer)->Stop();
	(*device_buffer)->Release();
	*device_buffer=NULL;
	
	/* This reclaims any contiguous space at the end of the array. At the end of
		the trial, this will have the effect of setting d_num_device_buffers
		back to 0 */
	i=FLX_MAX_PLAYBACK_BUFFERS-1;
	while(d_device_buffers[i].d_buffer==NULL) i--;
	if(i<d_num_device_buffers-1) d_num_device_buffers=i+1;

	flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Stopped DirectSound playback buffer "+flx_convert_to_string(device_buffer_descriptor));
  
} /* DirectSoundPlaybackDevice::stop */

/*****************************************************************************/

void DirectSoundCaptureDevice::stop(void){
	string cur_function="DirectSoundDevice::stop";

	if(!d_device_buffer) return;

	d_device_buffer->Stop();
	d_device_buffer->Release();
	d_device_buffer=NULL;
	flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Stopped DirectSound capture device");
  
} /* DirectSoundCaptureDevice::stop */

/*****************************************************************************/

bool DirectSoundDriver::init(void){
  string cur_function="DirectSoundDriver::init";
  
  if(!d_playback_device.init()){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Couldn't open DirectSound device for playback");
    return false;
  }

  if(!d_capture_device.init()){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Couldn't open DirectSound device for capture");
    return false;
  }

  flx_data->write_message(FLX_DATADEBUG,cur_function,"Initialized DirectSound driver");
  return true;

} /* DirectSoundDriver::init */

/*****************************************************************************/

void DirectSoundDriver::exit(void){
  string cur_function="DirectSoundDriver::exit";

  d_playback_device.exit();
  d_capture_device.exit();
  flx_data->write_message(FLX_DATADEBUG,cur_function,"Stopped DirectSound driver");

} /* DirectSoundDriver::exit */

/*****************************************************************************/

