//LIC// FLXLab v2.5 - A program for running psychology experiments.  
//LIC// Copyright (C) 2010 Todd R. Haskell (todd.haskell@wwu.edu) 
//LIC// 
//LIC// Use and distribution is governed by the terms of the 
//LIC// GNU General Public License. Certain portions of the 
//LIC// program may be subject to other licenses as well. See 
//LIC// the file LICENSE.TXT for details.
//LIC// 
#include <fstream>
#include <allegro.h>
#include <flxbase.h>
#include "flxtext\text_objects.h"
#include "flxtext\text_functions.h"
#include "flxtext\fonts.h"

typedef void (*textout_f)(BITMAP *,const FONT *,const char *,int,int,int,int);

textout_f textout_functions[] = { textout_ex,textout_centre_ex,textout_right_ex};

string *flx_default_text_font;
long *flx_default_text_size;
long *flx_default_text_justification;
FlxColor *flx_default_text_color;
FlxColor *flx_default_text_background_color;
bool *flx_default_antialiasing;

/*****************************************************************************/

bool FlxTextObject::update(void){
  /* We need to update our measurements of the size of the
     text before we actually draw it, so that alignment of
     text objects works correctly */
  string cur_function="FlxTextObject::update";

  /* first check to make sure the necessary attributes are defined */
  if(d_text && d_font && d_size && d_antialiasing){
    if(set_font_properties(*d_font,*d_size,*d_antialiasing)){
      d_text_width=get_text_width(*d_text);
      d_text_height=get_text_height();
      flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Updated text size values");
      return true;
    } else {
      flx_data->write_message(FLX_DATAERROR,cur_function,"Can't update font properties for text object '"+d_name+"'");
      return false;
    }
  } else {
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"FlxTextObject is not fully defined, skipping update.");
      return true;
  }

} /* FlxTextObject::update */

/*****************************************************************************/

/* If you draw antialiased text with a transparent background onto a
   bitmap that is also transparent, weird things happen (the
   transparent pixels in the bitmap are interpreted as actually being
   bright pink, so the text has a pinkish-purplish border). To avoid
   this, we use use the same algorithm as for drawing text with an
   opaque background, and "guess" what a reasonable background color
   would be, given the specified text color. This gives slightly
   sub-optimal results, but is better than not using antialiasing, and
   is the best we can do given that we can't always know what color
   the text is going to be written over. To get the best results,
   especially with colored text, use a TextBoxObject (rather than a
   FlxTextObject) and explicitly specify the color of the box, in which
   case that is used as the background color. */

int compute_aa_background_color(int fg_color){
  int bg_intensity;

  bg_intensity=255-(getr(fg_color)+getg(fg_color)+getb(fg_color))/3;
  return makecol(bg_intensity,bg_intensity,bg_intensity);

} /* compute_aa_background_color */

/*****************************************************************************/

void FlxTextObject::draw(void){
  string cur_function="FlxTextObject::draw";
  int background_color=0;
  FONT *font;

  flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Drawing text object '"+d_name+"' at ("+flx_convert_to_string(flx_pen->screen_x())+","+flx_convert_to_string(flx_pen->screen_y())+")");

  // get an Allegro font object with the appropriate properties
  set_font_properties(*d_font,*d_size,*d_antialiasing);
  font=get_allegro_font();

  if(d_antialiasing){
    /* If a specific non-transparent background color has been
       specified for the text, use that. If the transparent color
       has been specified for the background, do the antialiasing
       against the actual bitmap for the DisplayEvent. Otherwise,
       guess a reasonable background color. */
    if(d_background_color==flx_default_text_background_color){
      background_color=compute_aa_background_color(FlxTextObject::d_color->color_code());
    } else if(d_background_color==flx_transparent_color){
      background_color=-1;
    } else {
      background_color=d_background_color->color_code();
    }
  }
  textout_ex(flx_cur_buffer,font,d_text->c_str(),flx_pen->x(),flx_pen->y(),FlxTextObject::d_color->color_code(),background_color);

} /* FlxTextObject::draw */

/*****************************************************************************/

void FlxTextBoxObject::draw(void){
  string cur_function="TextBoxObject::draw";
  string *lines;
  int num_lines, i;
  int x_offset, y_offset;
  int background_color=0;
  FONT *font;

  flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Drawing "+flx_convert_to_string(x_size())+"x"+flx_convert_to_string(y_size())+" text box object '"+d_name+"' at ("+flx_convert_to_string(flx_pen->screen_x())+","+flx_convert_to_string(flx_pen->screen_y())+")");

  // get an Allegro font object with the appropriate properties
  set_font_properties(*d_font,*d_size,*d_antialiasing);
  font=get_allegro_font();

  num_lines=flx_text_wrap(*d_text,font,x_size()-2*text_height(font),lines);
  flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Split text into "+flx_convert_to_string(num_lines)+" lines");

  /* for convenience, we use side margins equal to the height of the current
     font */
   switch(*d_text_justification){
  case FLX_TEXT_JUSTIFICATION_LEFT:
    x_offset=text_height(font);
    break;
  case FLX_TEXT_JUSTIFICATION_CENTER:
    x_offset=x_size()/2;
    break;
  case FLX_TEXT_JUSTIFICATION_RIGHT:
    x_offset=x_size()-text_height(font);
    break;
  default:
    x_offset=0;
      flx_data->write_message(FLX_DATAERROR,cur_function,"Unknown text justification code");
  }
  y_offset=(y_size()-num_lines*text_height(font))/2;
  if(y_offset<0) y_offset=0;

  /* Determine what background color to use for drawing the text. If a
     specific non-transparent text background color has been
     specified, use that. If a transparent background color has been
     specified, then do antialiasing against the actual bitmap of the
     DisplayEvent. If no background color has been specified, then if
     a non-transparent color has been specified for the box, use
     that. Otherwise, guess a reasonable background color. */
  if(*d_antialiasing){
    if(d_background_color==flx_default_text_background_color){
      background_color=compute_aa_background_color(FlxTextObject::d_color->color_code());
    } else if(d_background_color==flx_transparent_color){
      background_color=-1;
    } else {
      background_color=d_background_color->color_code();
    }
  }
  for(i=0;i<num_lines;i++){
    (*(textout_functions[*d_text_justification]))(flx_cur_buffer,font,lines[i].c_str(),flx_pen->x()+x_offset,flx_pen->y()+y_offset,d_color->color_code(),background_color);
    y_offset+=text_height(font);
  }

} /* FlxTextBoxObject::draw */

/*****************************************************************************/

bool NewTextObject(string *name,string *text){
  string cur_function="New"+flx_cur_script_command;
  FlxGraphicsObject *new_object;

  flx_data->write_message(FLX_DATASCRIPT,cur_function,"Creating new "+flx_cur_script_command+" '"+*name+"'");
  if(flx_cur_script_command=="TextObject"){
    new_object=new FlxTextObject(*name,text);
  } else if(flx_cur_script_command=="TextBoxObject"){
    new_object=new FlxTextBoxObject(*name,text);
  } else {
    flx_data->write_message(FLX_DATAERROR,cur_function,"NewTextObject called via unknown name "+flx_cur_script_command);  
    return false;
  }
  return true;

} /* NewTextObject */

/*****************************************************************************/

void text_object_init_callback(void){

  flx_default_text_font=flx_string("URWNimbus");
  flx_default_text_size=flx_long(48);
  flx_default_text_justification=flx_long(FLX_TEXT_JUSTIFICATION_LEFT);
  flx_default_text_color=flx_color(0,0,0);
  flx_default_text_background_color=flx_color(1,2,3);
  flx_default_antialiasing=flx_bool(true);

} /* text_object_init_callback */

/*****************************************************************************/

void flx_text_objects_init(void){

  flx_add_command("TextObject",NewTextObject);
  flx_add_command("TextBoxObject",NewTextObject);
  
} /* flx_text_objects_init */

/*****************************************************************************/

