//LIC// FLXLab v2.5 - A program for running psychology experiments.  
//LIC// Copyright (C) 2010 Todd R. Haskell (todd.haskell@wwu.edu) 
//LIC// 
//LIC// Use and distribution is governed by the terms of the 
//LIC// GNU General Public License. Certain portions of the 
//LIC// program may be subject to other licenses as well. See 
//LIC// the file LICENSE.TXT for details.
//LIC// 
#include "flxbase\scopes.h"
#include "flxbase\FlxDataSystem.h"
#include "flxbase\scripts.h"
#include "flxbase\command_manager.h"
#include "flxbase\string_utils.h"
#include "flxbase\event_hooks.h"
#include "flxbase\event_manager.h"
#include "flxbase\FlxObject.h"

FLX_LINKAGE FlxObjectList flx_object_list;

/*****************************************************************************/

FlxObject **FlxObjectList::find_name(const string &name,bool advance){
  /* Returns a pointer to the first item in the list which has a name
     matching <name>, starting from <d_cur_item>. If <advance> is
     true, also makes any matching item the current item. */
  FlxObjectListItem *index;

  index=static_cast<FlxObjectListItem *>(d_cur_item);
  while(index!=NULL){
    if(index->d_item->name()==name){
      if(advance) d_cur_item=index;
      return &(index->d_item);
    }
    index=static_cast<FlxObjectListItem *>(index->d_next);
  }
  return NULL;

} /* FlxObjectList::find_name */

/*****************************************************************************/

FlxObject **FlxObjectList::find_scope(const string &scope,bool advance){
  /* Returns a pointer to the first item in the list which has a scope
     matching <scope>, starting from <d_cur_item>. If <advance> is
     true, also makes any matching item the current item. */
  FlxObjectListItem *index;

  index=static_cast<FlxObjectListItem *>(d_cur_item);
  while(index!=NULL){
    if(index->d_item->scope()==scope){
      if(advance) d_cur_item=index;
      return &(index->d_item);
    }
    index=static_cast<FlxObjectListItem *>(index->d_next);
  }
  return NULL;

} /* FlxObjectList::find_scope */

/*****************************************************************************/

void FlxObjectList::dump_list(void){
  FlxObjectListItem *index;

  index=static_cast<FlxObjectListItem *>(d_list);
  while(index!=NULL){
    index->d_item->dump_object();
    index=static_cast<FlxObjectListItem *>(index->d_next);
  }

} /* FlxObjectList::dump_list */

/*****************************************************************************/

FlxObject::FlxObject(const string &name): d_name(name), d_scope(flx_cur_scope_tag) {
  string cur_function="FlxObject::FlxObject";

  // check if an object with this name already exists
  if(flx_object_list.find_name(name)){
        flx_data->write_message(FLX_DATAERROR,cur_function,"Can't create new object, an object named '"+name+"' already exists.");
  } else {
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Inserting object '"+name+"' into the object list.");
    flx_object_list.insert_at_beginning(this); 
  }

} /* FlxObject::FlxObject */

/*****************************************************************************/
/* This ensures that there won't be any bad pointers to the object
   floating around in a list of sources or dependents. It won't solve
   the problem of bad pointers if some other object stores a pointer
   to this one somewhere other than its list of sources and
   dependents. However, this is unlikely to happen due to our scoping
   rules (a source may not have a narrower scope than a dependent). */
FlxObject::~FlxObject(void){
  FlxObject **foh;
  
  while((foh=d_sources.item())){
    (*foh)->remove_dependent(this);
    d_sources.advance();
  }
  d_sources.rewind();

  while((foh=d_dependents.item())){
    (*foh)->remove_source(this);
    d_dependents.advance();
  }
  d_dependents.rewind();

} /* FlxObject::~FlxObject */

/*****************************************************************************/

void FlxObject::remove_dependent(FlxObject *dependent){
  string cur_function="remove_dependent";

  d_dependents.advance_to_item(dependent);
  d_dependents.remove();
  d_dependents.rewind();

} /* FlxObject::remove_dependent */

/*****************************************************************************/

void FlxObject::remove_source(FlxObject *source){
  string cur_function="remove_source";

  d_sources.advance_to_item(source);
  d_sources.remove();
  d_sources.rewind();

} /* FlxObject::remove_source */

/*****************************************************************************/

/* This function adds the object itself as well as all its dependents
	to <update_list>.  If the object is already in <update_list>,
	it is moved to the end of the list. */
void FlxObject::add_to_update_list(FlxLinkedList<FlxObject *> &update_list){
  string cur_function="add_to_update_list";
  FlxObject **object_handle;
  bool dependents_updated=false;
  
  flx_data->write_message(FLX_DATADDEBUG,cur_function,"Adding '"+this->name()+"' to update list");
  /* First add the object itself */
  while((object_handle=update_list.item())){
    /* check if the current object is already in the list; if
       so, remove it (it will be re-inserted at the end */
    if((*object_handle)==this){
      update_list.remove();
    }
    update_list.advance();
  }
  update_list.insert(this);
  update_list.rewind();

  /* Next, add any dependents */
  while((object_handle=d_dependents.item())){
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Adding dependent of '"+this->name()+"' named '"+(*object_handle)->name()+"' to update list");
    (*object_handle)->add_to_update_list(update_list);    
      dependents_updated=true;
      d_dependents.advance();
  }
  d_dependents.rewind();
  if(!dependents_updated) flx_data->write_message(FLX_DATADDDEBUG,cur_function,"'"+this->name()+"' has no dependents to update"); 
  
} /* FlxObject::flx_add_to_update_list */

/*****************************************************************************/

bool FlxObject::update_object_and_dependents(void){
  string cur_function="update_object_and_dependents";
  FlxLinkedList<FlxObject *> update_list;
  FlxObject **cur_object;
  bool return_val=true;

  this->add_to_update_list(update_list);
  while((cur_object=update_list.item())){
    flx_data->write_message(FLX_DATADDEBUG,cur_function,"Updating '"+(*cur_object)->name()+"'");
    return_val=((*cur_object)->update() ? return_val : false);
    update_list.advance();
  }
  update_list.rewind();
  return return_val;

} /* FlxObject::update_object_and_dependents */

/*****************************************************************************/

bool FlxObject::is_static(void){
  string cur_function="FlxObject::is_static";
  FlxObject **foh;
  bool return_val=true;

  flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Checking if "+d_name+" is static");
  if(d_name=="root"){
    return false;
  } else {
    while((foh=d_sources.item())){
      flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Checking source "+(*foh)->name());
      if(!(*foh)->is_static()){
	return_val=false;
	break;
      }
      d_sources.advance();
    }
    d_sources.rewind();
    return return_val;
  }
  
} /* FlxObject::is_static */

/*****************************************************************************/

void FlxObject::dump_object(void){
  string cur_function="FlxObject::dump_object";
  FlxObject **foh;

  flx_data->write_message(FLX_DATAINFO,cur_function,"Object: "+d_name+" Scope: "+d_scope+" Type: "+typeid(*this).name());
  while((foh=d_dependents.item())){
    flx_data->write_message(FLX_DATAINFO,cur_function,"  Dependent: "+(*foh)->name());
    d_dependents.advance();
  }
  d_dependents.rewind();
  while((foh=d_sources.item())){
    flx_data->write_message(FLX_DATAINFO,cur_function,"  Source: "+(*foh)->name());
    d_sources.advance();
  }
  d_sources.rewind();

} /* FlxObject::dump_object */

/*****************************************************************************/

