//LIC// FLXLab v2.5 - A program for running psychology experiments.  
//LIC// Copyright (C) 2010 Todd R. Haskell (todd.haskell@wwu.edu) 
//LIC// 
//LIC// Use and distribution is governed by the terms of the 
//LIC// GNU General Public License. Certain portions of the 
//LIC// program may be subject to other licenses as well. See 
//LIC// the file LICENSE.TXT for details.
//LIC// 
#include <fstream>
#include <iomanip>
#include "flxbase\FlxDataSystem.h"
#include "flxbase\flxbase.h"
#include "flxbase\strings.h"
#include "flxbase\logical_and_numeric_types.h"
#include "flxbase\command_manager.h"
#include "flxbase\scripts.h"
#include "flxbase\event_hooks.h"
#include "flxbase\time.h"
#include "flxbase\file_system.h"
#include "flxbase\FlxStack.h"

int num_user_data_files;
FlxStack<int> context_stack;

/*****************************************************************************/

bool UseDataFile(string *filename){
  string cur_function="UseDataFile";

  flx_data->write_message(FLX_DATASCRIPT,cur_function,"Using data file '"+*filename+"'");
  /* If the file associated with the current data queue is already
     open, then create a new data queue associated with the new
     filename. If the data file associated with the current data queue
     has not been opened yet, then just set the name of the file to be
     associated with the existing queue. */
  if(flx_data->file_is_open()){
    flx_data->new_queue(*filename);
    num_user_data_files++;
  } else {
    flx_data->use_file(*filename);
  }
  return true;

} /* UseDataFile */

/*****************************************************************************/
/* If the user is not careful about creating and closing data files,
it is very easy to leave files open after they are no longer needed,
or to close too many data files. To provide some protection against
that, FLXLab uses the notion of a "context", which is similar to (but
not the same as) a scope. In practice, a context is generally equivalent
to a particular script. There are two rules concerning closing data files:

1) Any data files created within a particular context which are not
explicitly closed by a script with CloseDataFile will be automatically
closed when that context ends.

2) You can only use the CloseDataFile command to close data files that
were created within the current context.

These two rules are implemented by the next three functions.
*/

bool CloseDataFile(void){
  string cur_function="CloseDataFile";
  int *num_files_before_context;

  flx_data->write_message(FLX_DATASCRIPT,cur_function,"Closing current data file");
  if(!(num_files_before_context=context_stack.read())){
    flx_data->write_message(FLX_DATAERROR,cur_function,"CloseDataFile command called when no contexts exist");
    return false;
  }

  if(num_user_data_files>*num_files_before_context){
    flx_data->delete_queue();
    num_user_data_files--;
    return true;
  } else {
    flx_data->write_message(FLX_DATAERROR,cur_function,"The CloseDataFile command can only be used to close data files created within the current script");
    return false;
  }

} /* CloseDataFile */

/*****************************************************************************/

void begin_data_file_context(void){
  string cur_function="begin_data_file_context";

  flx_data->write_message(FLX_DATADEBUG,cur_function,"Beginning data file context");
  context_stack.push(num_user_data_files);

} /* begin_data_file_context */

/*****************************************************************************/


void end_data_file_context(void){
  string cur_function="end_data_file_context";
  int num_files_before_context;

  /* the stack should never be empty when this is called, but check
     just in case */
  if(!context_stack.read()){ 
    flx_data->write_message(FLX_DATAERROR,cur_function,"end_data_file_context called when no contexts exist");
    return;
  }

  flx_data->write_message(FLX_DATADEBUG,cur_function,"Ending data file context; closing any open data files created by the current script");
  num_files_before_context=context_stack.pop();
  while(num_user_data_files>num_files_before_context){
    flx_data->delete_queue();
    num_user_data_files--;    
  }

} /* end_data_file_context */

/*****************************************************************************/

bool RecordToDataFile(string *level_string,bool *flag){
  string cur_function="RecordToDataFile";
  unsigned int i;

  for(i=0;i<FLX_DATA_LEVELS;i++){
    if(*level_string==FlxDataQueue::level_labels[i]){
      flx_data->write_message(FLX_DATASCRIPT,cur_function,"Turning recording of "+FlxDataQueue::level_labels[i]+" messages "+(*flag ? "on" : "off"));
      flx_data->queue()->set_record_to_file_flag(i,*flag);
      return true;
    }
  }
  flx_data->write_message(FLX_DATAERROR,cur_function,"Unknown message type '"+*level_string+"'");
  return false;

} /* RecordToDataFile */

/*****************************************************************************/

bool DataTimeStamp(bool *flag){
  string cur_function="DataTimeStamp";

  flx_data->write_message(FLX_DATASCRIPT,cur_function,"Setting timestamping of data messages to "+(*flag ? string("on") : string("off")));
  flx_data->queue()->set_do_time_stamp(*flag);
  return true;

} /* DataTimeStamp */

/*****************************************************************************/

bool BufferData(bool *flag){
  string cur_function="BufferData";

  flx_data->write_message(FLX_DATASCRIPT,cur_function,"Turning buffering of data messages "+(*flag ? string("on") : string("off")));
  flx_data->queue()->set_buffer_messages(*flag);
  return true;

} /* BufferData */

/*****************************************************************************/

bool AppendData(bool *flag){
  string cur_function="AppendData";

  flx_data->write_message(FLX_DATASCRIPT,cur_function,"Setting mode for writing data files to "+(*flag ? string("append") : string("overwrite")));
  flx_data->queue()->set_append_to_existing_file(*flag);
  return true;

} /* AppendData */

/*****************************************************************************/

bool flush_data_queue(void){

  flx_data->queue()->flush();
  return true;

} /* flush_data_queue */

/*****************************************************************************/

void flx_data_manager_init(void){

  flx_add_command("UseDataFile",UseDataFile);
  flx_add_command("CloseDataFile",CloseDataFile);
  flx_add_command("RecordToDataFile",RecordToDataFile,"true");
  flx_add_command("DataTimeStamp",DataTimeStamp,"true");
  flx_add_command("BufferData",BufferData,"true");
  flx_add_command("AppendData",AppendData,"true");

  flx_add_event_hook("flush_data_queue",flush_data_queue,FLX_EVENT_HOOK_PRESESSION);
  flx_add_event_hook("flush_data_queue",flush_data_queue,FLX_EVENT_HOOK_POSTTRIAL);
  flx_add_event_hook("flush_data_queue",flush_data_queue,FLX_EVENT_HOOK_POSTSESSION,FLX_EVENT_HOOK_LATE);

  num_user_data_files=0;

} /* flx_data_manager_init */

/*****************************************************************************/

