//LIC// FLXLab v2.5 - A program for running psychology experiments.  
//LIC// Copyright (C) 2010 Todd R. Haskell (todd.haskell@wwu.edu) 
//LIC// 
//LIC// Use and distribution is governed by the terms of the 
//LIC// GNU General Public License. Certain portions of the 
//LIC// program may be subject to other licenses as well. See 
//LIC// the file LICENSE.TXT for details.
//LIC// 
#include <fstream>
#include <sstream>
#include "flxbase\logical_and_numeric_types.h"
#include "flxbase\strings.h"
#include "flxbase\scripts.h"
#include "flxbase\command_manager.h"
#include "flxbase\FlxDataSystem.h"
#include "flxbase\event_hooks.h"
#include "flxbase\condition_manager.h"
#include "flxbase\object_manager.h"
#include "flxbase\FlxTriplexList.h"
#include "flxbase\scopes.h"
#include "flxbase\file_system.h"
#include "flxbase\OS_specific_file_system.h"
#include "flxbase\random.h"
#include "flxbase\FlxSetting.h"
#include "flxbase\FlxStimulusList.h"

/* We need this so we don't have to qualify all STL names with
"std::" */
using namespace std;

/*****************************************************************************/

FLX_LINKAGE bool flx_type_convert(const string &s,FlxStimulusList *&slp){

  return flx_object_convert(s,slp,"FlxStimulusList");
} 

/*****************************************************************************/

bool NewStimulusList(string *list_name,string *list_file){
  string cur_function="NewStimulusList";
  string expanded_list_path;
  FlxStimulusList *new_list;
  ifstream *list_stream;
  
  expanded_list_path=flx_expand_filename(*list_file);
  if(expanded_list_path.empty()){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+":Can't find list file '"+*list_file+"'");
    return false;
  } else {
    list_stream=new ifstream(expanded_list_path.c_str());
    if(!*list_stream){
      flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": Couldn't open list file '"+*list_file+"'");
      
      return false;
    } else {
      new_list=new FlxStimulusList(*list_name);
      flx_add_object_source(new_list,flx_dependency_root);
      if(new_list->read_from_stream(*list_stream)){
	flx_data->write_message(FLX_DATASCRIPT,cur_function,"Created stimulus list "+*list_name+" from file '"+*list_file+"'");
	flx_change_global_setting(&flx_cur_list,new_list,"flx_cur_list");
	return true;
      } else {
	flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": Error while reading stimulus list");
	delete new_list;
	return false;
      }
    }
  }
    
} /* NewStimulusList */

/*****************************************************************************/

bool LabelListColumn(long *col,string *label){
  string cur_function="LabelListColumn";
  FlxObject *fop;

  if(!flx_cur_list){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": No stimulus list is currently selected");
    return false;
  }

  if(*col<1 || *col>flx_cur_list->num_fields()){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": Column number must be between 1 and "+flx_convert_to_string(flx_cur_list->num_fields()));
    return false;
  } else {
    fop=flx_create_scalar_object(*label,flx_cur_list->field_pointer((*col)-1));
    flx_add_object_source(fop,flx_cur_list);
    flx_data->write_message(FLX_DATASCRIPT,cur_function,"Labeling column "+flx_convert_to_string(*col)+" of list '"+flx_cur_list->name()+"' with label of '"+*label+"'");
    return true;
  }

} /* LabelListColumn */

/*****************************************************************************/

bool SelectList(FlxStimulusList *slp){
  string cur_function="SelectList";

  flx_data->write_message(FLX_DATASCRIPT,cur_function,"Selecting stimulus list '"+slp->name()+"'");
  flx_change_global_setting(&flx_cur_list,slp,"flx_cur_list");
  return true;

} /* SelectList */

/*****************************************************************************/

bool ResetList(void){
  string cur_function="ResetList";

  if(flx_cur_list==NULL){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Can't reset list, no list currently selected.");
	return false;
  } else {
    flx_data->write_message(FLX_DATASCRIPT,cur_function,"Resetting '"+flx_cur_list->name()+"'");
    flx_cur_list->reset();
  }
  return true;

} /* ResetList */

/*****************************************************************************/

bool RandomizeList(void){
  string cur_function="RandomizeList";

  if(flx_cur_list==NULL){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Can't randomize list, no list currently selected.");
	return false;
  } else {
    flx_data->write_message(FLX_DATASCRIPT,cur_function,"Randomizing '"+flx_cur_list->name()+"'");
    flx_cur_list->randomize();
  }
  return true;

} /* RandomizeList */

/*****************************************************************************/

bool JoinLists(string *list_name,FlxLiteralString *s){
  string cur_function="JoinLists";
	istringstream lists_to_join(s->c_str());
	string next_list;
	FlxStimulusList *list_ptr, *combined_list;
	FlxLinkedList<FlxStimulusList *> the_lists;

	flx_data->write_message(FLX_DATASCRIPT,cur_function,"Joining lists to create new list '"+*list_name+"'");
	/* This code parses the string <s> into white-space-separated
		fields, and then tries to convert each such field to a
		FlxStimulusList *, which is then stored in a
		LinkedList. */
	while(lists_to_join >> next_list && !lists_to_join.fail()){
		list_ptr=NULL;
		flx_type_convert(next_list,list_ptr);
		if(list_ptr==NULL){
			    flx_data->write_message(FLX_DATAERROR,cur_function,"Can't join list '"+next_list+"', list doesn't exist.");
				return false;
		} else {
			the_lists.insert(list_ptr);
		}
	}
	/* Make sure at least one list was provided */
	if(the_lists.item()==NULL){
	    flx_data->write_message(FLX_DATAERROR,cur_function,"Attempting to join lists, but the set of lists to join is empty.");
		return false;
	} else {
		// create the actual combined list
		combined_list=new FlxStimulusList(*list_name);
		flx_add_object_source(combined_list,flx_dependency_root);
		if(combined_list->join_lists(the_lists)){
		  flx_change_global_setting(&flx_cur_list,combined_list,"flx_cur_list");
		  return true;
		} else {
		  flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": Error while joining stimulus lists.");
		  return false;
		}
	}

} /* JoinLists */

/*****************************************************************************/

class ListCondition : public FlxCondition {
public:
  ListCondition(const string &name) : FlxCondition(name) {}
  ~ListCondition(void) {}
  bool evaluate(void){ if(flx_cur_list) return flx_cur_list->end_of_list(); else return true; } 
};

/*****************************************************************************/

FlxCondition *list_condition_parser(const string &keyword,int arg_count,string *arg_array){
  string cur_function="list_condition_parser";
  string name;

  if(arg_count!=1 || arg_array[0]!="end"){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": Bad list condition");
    return NULL;
  } 

  name=flx_unique_name();
  flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Creating new list condition named '"+name+"'");
  return new ListCondition(name);

} /* list_condition_parser */

/*****************************************************************************/

void flx_list_manager_init(void){

  flx_add_command("StimulusList",NewStimulusList);
  flx_add_command("LabelListColumn",LabelListColumn);
  flx_add_command("SelectList",SelectList);
  flx_add_command("ResetList",ResetList);
  flx_add_command("RandomizeList",RandomizeList);
  flx_add_command("JoinLists",JoinLists);
  flx_add_condition_type("list",list_condition_parser);

  flx_cur_list=NULL;

} /* flx_list_manager_init */

/*****************************************************************************/

