//LIC// FLXLab v2.5 - A program for running psychology experiments.  
//LIC// Copyright (C) 2010 Todd R. Haskell (todd.haskell@wwu.edu) 
//LIC// 
//LIC// Use and distribution is governed by the terms of the 
//LIC// GNU General Public License. Certain portions of the 
//LIC// program may be subject to other licenses as well. See 
//LIC// the file LICENSE.TXT for details.
//LIC// 
#include "flxbase\FlxDataSystem.h"
#include "flxbase\FlxStack.h"
#include "flxbase\FlxTriplexList.h"
#include "flxbase\command_manager.h"
#include "flxbase\scopes.h"

FLX_LINKAGE string flx_cur_scope_tag;
FlxStack<string> scope_stack;
FlxDuplexList<FlxLinkedList<string> *,string> scope_hierarchy;
FlxTriplexList<FlxScopeCleanupHook,string,string> scope_cleanup_hooks;
FlxTriplexList<FlxScopeExitHook,string,string> scope_exit_hooks;

/*****************************************************************************/

FLX_LINKAGE void flx_add_scope_cleanup_hook(const string &label,FlxScopeCleanupHook h){
  string cur_function="flx_add_scope_cleanup_hook";

  flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Adding scope cleanup hook '"+label+"'");
  scope_cleanup_hooks.insert(h,label,flx_cur_scope_tag);

} /* flx_add_scope_cleanup_hook */

/*****************************************************************************/

FLX_LINKAGE void flx_add_scope_exit_hook(const string &label,FlxScopeExitHook h){
  string cur_function="flx_add_scope_exit_hook";

  flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Adding exit hook for scope '"+flx_cur_scope_tag+"'");
  scope_exit_hooks.insert(h,label,flx_cur_scope_tag);

} /* flx_add_scope_exit_hook */

/*****************************************************************************/

FLX_LINKAGE bool is_nested_scope(const string &tag1,const string &tag2){
  /* Returns <true> if the scope <tag1> is nested inside the scope <tag2> */
  string cur_function="is_nested_scope";
  FlxLinkedList<string> **nested_scopes;

  flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Checking if scope '"+tag1+"' is nested inside scope '"+tag2+"'");

  /* make sure tag1 is a real scope */
  if(!scope_hierarchy.find_item2(tag1)){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Error when checking for nested scope, '"+tag1+"' isn't in scope hierarchy");
    return false;
  }

  /* make sure tag2 is a real scope */
  if(!(nested_scopes=scope_hierarchy.find_item2(tag2))){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Error when checking for nested scope, '"+tag2+"' isn't in scope hierarchy");
    return false;
  }
  
  /* check to see if tag1 is nested inside tag2 */
  if((*nested_scopes)->find_item(tag1)){
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Is nested");
    return true;
  } else {
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Not nested");
    return false;
  }

} /* is_nested_scope */

/*****************************************************************************/

FLX_LINKAGE bool flx_begin_scope(const string &tag){
  string cur_function="flx_begin_scope";
  FlxLinkedList<string> *nested_scopes;

  flx_data->write_message(FLX_DATADEBUG,cur_function,"Beginning scope '"+tag+"'");

  /* make sure this scope doesn't exist yet */
  if(scope_hierarchy.find_item2(tag)){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Can't create new scope, a scope named '"+tag+"' already exists.");
    return false;
  }

  /* add the new scope to the list of nested scopes for the current scope */
  flx_data->write_message(FLX_DATADEBUG,cur_function,"Adding '"+tag+"' to the nested scopes list for '"+flx_cur_scope_tag+"'");
  /* We add the scope at the beginning of the list so that the last scope
     added is the first scope ended */
  (*(scope_hierarchy.find_item2(flx_cur_scope_tag)))->insert_at_beginning(tag);
  /* We explicitly rewind the list here because if the current item is the 
     first item in the list, then inserting something at the beginning has 
     the effect of making it the second item in the list */
  (*(scope_hierarchy.find_item2(flx_cur_scope_tag)))->rewind();

  // add the new scope to the scope hierarchy
  nested_scopes=new FlxLinkedList<string>();
  scope_hierarchy.insert(nested_scopes,tag);

  // update the scope stack
  scope_stack.push(flx_cur_scope_tag);
  flx_cur_scope_tag=tag;
  return true;

} /* flx_begin_scope */

/*****************************************************************************/

FLX_LINKAGE void flx_end_scope(void){
  string cur_function="flx_end_scope";

  flx_data->write_message(FLX_DATADEBUG,cur_function,"Ending scope '"+flx_cur_scope_tag+"'");
  flx_cur_scope_tag=scope_stack.pop();


} /* flx_end_scope */

/*****************************************************************************/

FLX_LINKAGE bool flx_delete_scope(const string &tag){
  string cur_function="flx_delete_scope";
  FlxLinkedList<string> **nested_scopes;
  string *nested_scope_tag;
  FlxScopeCleanupHook *hook;

  flx_data->write_message(FLX_DATADEBUG,cur_function,"Preparing to delete scope "+tag);
  /* make sure the scope exists before trying to delete it */
  if(!(nested_scopes=scope_hierarchy.find_item2(tag))){
       flx_data->write_message(FLX_DATAERROR,cur_function,"Can't delete scope '"+tag+"', there is no scope with this name.");
    return false;
  } 

  /* scope exists, delete any nested scopes */
  while((nested_scope_tag=(*nested_scopes)->item())){
    /* Since the user can force a scope to be deleted with the
     EndScope command, we have to check if each nested 
    scope still exists before trying to delete it. */
    if(scope_hierarchy.find_item2(*nested_scope_tag)){
      flx_delete_scope(*nested_scope_tag);
    }
    (*nested_scopes)->remove();
  }
  delete (*nested_scopes);
  /* we can't advance the current item to the scope tag earlier, because
     cleaning up the nested scopes might change the current item */
  scope_hierarchy.advance_to_item2(tag);
  scope_hierarchy.remove();
  scope_hierarchy.rewind();

  flx_data->write_message(FLX_DATADEBUG,cur_function,"Deleting scope '"+tag+"'");

  /* call scope cleanup hooks, removing any that are associated with
     the scope we're ending */
  while((hook=scope_cleanup_hooks.item())){
    (**hook)(tag);
    if(*(scope_cleanup_hooks.item3())==tag){
      flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Removing scope cleanup hook '"+*(scope_cleanup_hooks.item2())+"'");
      scope_cleanup_hooks.remove();
    } else {
      scope_cleanup_hooks.advance();
    }
  }
  scope_cleanup_hooks.rewind();
    
  // call & remove any scope-specific exit hooks
  while(scope_exit_hooks.advance_to_item3(tag)){
    flx_data->write_message(FLX_DATADDEBUG,cur_function,"Calling and removing exit hook '"+*(scope_exit_hooks.item2())+"' for scope '"+tag+"'");
    (*(scope_exit_hooks.item()))();
    scope_exit_hooks.remove();
  }
  scope_exit_hooks.rewind();

  // if we're ending the current scope, update the scope tag
  if(tag==flx_cur_scope_tag){
    flx_cur_scope_tag=scope_stack.pop();
  }

  return true;

} /* flx_delete_scope */

/*****************************************************************************/

bool BeginScope(string *name){
  string cur_function="BeginScope";
  
  flx_data->write_message(FLX_DATASCRIPT,cur_function,"Beginning new scope '"+*name+"'");
  return flx_begin_scope(*name);

} /* BeginScope */

/*****************************************************************************/

/* This command actually both ends & deletes the current scope */
bool EndScope(string *name){
  string cur_function="EndScope";
  string scope_name;

  flx_data->write_message(FLX_DATASCRIPT,cur_function,"Ending and deleting scope '"+*name+"'");
  flx_end_scope();
  /* We can't pass <*name> to <flx_delete_scope> directly because the
     <tag> argument to <flx_delete_scope> is passed by reference, meaning
     that <tag> will only be defined for as long as <name> is a valid
     pointer.  However, ending the scope will actually free the memory
     associated with <name>. Once this happens, the value of <tag> 
     inside <flx_delete_scope> becomes undefined, and the function will
     crash. The solution is to make a local copy of the scope name we
     want to end, and pass that to <flx_delete_scope> */
  scope_name=*name;
  return flx_delete_scope(scope_name);

} /* EndScope */

/*****************************************************************************/

bool DumpScopes(void){
  string cur_function="DumpScopes";
  string *scope_tag;
  FlxLinkedList<string> **nested_scopes;
  
  while((scope_tag=scope_hierarchy.item2())){
    flx_data->write_message(FLX_DATAINFO,cur_function,"Scope: "+*scope_tag);
    nested_scopes=scope_hierarchy.item();
    while((scope_tag=(*nested_scopes)->item())){
      flx_data->write_message(FLX_DATAINFO,cur_function,"  Nested scope: "+*scope_tag);
      (*nested_scopes)->advance();
    }
    (*nested_scopes)->rewind();
    scope_hierarchy.advance();
  }
  scope_hierarchy.rewind();
  return true;

} /* DumpScopes */

/*****************************************************************************/

void flx_scopes_init(void){
  string cur_function="scopes_init";

  flx_data->write_message(FLX_DATADEBUG,cur_function,"Beginning scope 'BASE'");
  scope_hierarchy.insert(new FlxLinkedList<string>,flx_cur_scope_tag="BASE");

  flx_add_command("BeginScope",BeginScope);
  flx_add_command("EndScope",EndScope);
  flx_add_command("DumpScopes",DumpScopes);

} /* flx_scopes_init */

/*****************************************************************************/

