/*
 * Glyph Keeper example - font comparison program
 *
 * Copyright (c) 2003-2007 Kirill Kryukov
 *
 * This file is part of Glyph Keeper library, and may only be used,
 * modified, and distributed under the terms of the Glyph Keeper
 * license, located in the file 'license.txt' within this package.
 */


/*
 * This is a font comparison program. It shows character charts rendered with
 * two or more fonts, so you can compare the fonts visually.
 *
 * Running: 
 *     compare_alleg <font1> [<font2>] ... [<fontN>]
 * Up to 10 fonts can be used.
 *
 * Controlling:
 *     Up/Down             - scroll one row
 *     Shift + Up/Down     - fast scroll
 *     PgUp/PgDn           - page up/down
 *     Left/Right          - scroll one character
 *     Shift + Left/Right  - fast scroll
 *     ',' / '.'           - change font size
 *     '[' / ']'           - change cell size
 *     1/2                 - change angle
 *     3/4                 - change italic/anti-italic angle
 *     5/6                 - change width/height ratio
 *     7/8                 - change boldness strength
 *     Home                - change angle, italics, ratio and boldness back to normal
 *     A                   - toggle monochrome/antialiased
 *     F                   - toggle freetype auto-hinting on/off
 *     T                   - toggle default (truetype) hinting on/off
 *     G                   - toggle grid
 *     C                   - toggle clipping
 *     R                   - toggle center crosshair on/off
 *     F1                  - help (TO DO)
 *     F11                 - make screenshot
 *     Esc                 - exit
 */

#include <stdlib.h>
#include <stdio.h>
#include <allegro.h>
#include "glyph.h"

FILE *logfile = 0;

char *config_file = "compare_alleg.cfg";

/* Maximum number of fonts this program can compare */
enum { max_face_num = 10 };

int screen_width = 0;
int screen_height = 0;
int color_depth = 0;

int glyph_cache_size = 40*1024*1024;
int use_buffer = 1;
int show_grid = 1;
int text_size = 25;
int cell_size = 36;
int italic = 0;
int bold = 0;
int antialiasing_on = 1;
int ft_hinting_on = 0;
int tt_hinting_on = 1;
int clipping_on = 1;
int crosshair_on = 0;

int allegro_installed = 0;

int face_num = 0;
char *default_font_file = "../fonts/DejaVuSans.ttf";
const char* font_search_path = "";
char *font_files[max_face_num];
GLYPH_FACE *faces[max_face_num];
GLYPH_KEEP *keep = 0;
GLYPH_REND *rends[max_face_num];
BITMAP *pattern_bitmap = 0;
BITMAP *buffer = 0;
int back_color,c1,c2;
volatile unsigned csecs=0;
void counter(void) { csecs++; }
END_OF_FUNCTION(counter)
int start_code = 32;
int cells_in_row;
int cells_in_col;
enum { num_angles = 120 }; /* 3 degrees increment */
int angle_index = 0;
double angles[num_angles];
int grid_color, crosshair_color;
enum { num_colors = 11 };
int colors_html[num_colors] = {0x39EF86,0x18A2DC,0xEFE939,0xB15BDD,0x95DD5B,
                               0xb03a3a,0x40d1c5,0xe1aa34,0x5d47e4,0x9ae3d8,
                               0x000000};
int colors[num_colors];
int min_cell_size = 8;
int max_cell_size = 200;
enum { num_ratios = 50 };
int ratio_index = 0;
double ratios[num_ratios*2+1];


void messenger(const char* const msg)
{
    if (logfile) fprintf(logfile,"%s",msg);
}


void read_config()
{
    fprintf(logfile,"Reading configuration file \"%s\"\n",config_file);
    set_config_file(config_file);
    screen_width = get_config_int(0,"screen_width",800);
    screen_height = get_config_int(0,"screen_height",600);
    color_depth = get_config_int(0,"color_depth",0);
    font_search_path = get_config_string(0,"font_search_path","");
    if (color_depth!=0 && color_depth!=8 && color_depth!=15 && color_depth!=16 && color_depth!=24 && color_depth!=32)
        color_depth = 0;
    fprintf(logfile,"Graphics mode: %d x %d, %d bit\n",screen_width,screen_height,color_depth);
}


void update()
{
    int face_index;
    int status_line_height = 12;
    int face_area_top = 20;
    int face_area_width;
    int unused_width;
    int left_margin;
    int face_area_height;
    int top_margin;

    acquire_bitmap(buffer);

    rectfill(buffer,0,0,buffer->w-1,face_area_top-1,back_color);
    textprintf_ex(buffer,font,1,1,c1,back_color,"%s rendering, %s hinting   glyph cache: %d bytes",
        antialiasing_on?"antialiased":"monochrome",
        tt_hinting_on?"default":ft_hinting_on?"freetype":"no",
        gk_keeper_byte_count(keep));

    cells_in_row = (screen_width-face_num)/(face_num*(cell_size+1));
    face_area_width = cells_in_row*(cell_size+1)+1;
    unused_width = screen_width - face_num*face_area_width;
    left_margin = unused_width/(2*face_num);
    cells_in_col = (screen_height-face_area_top-status_line_height-face_num)/(cell_size+1);
    face_area_height = cells_in_col*(cell_size+1)+1;
    top_margin = (screen_height-face_area_top-status_line_height-face_area_height)/2;

    for (face_index = 0; face_index < face_num; face_index++)
    {
        GLYPH_REND *rend = rends[face_index];
        int xleftborder = screen_width*face_index/face_num;
        int xrightborder = screen_width*(face_index+1)/face_num-1;
        int xleft = xleftborder + left_margin;
        int xright = xleft + face_area_width - 1;
        int ytop = face_area_top+top_margin;
        int ybottom = ytop + face_area_height - 1;
        int i,col,row,x,y,now_code;

        /* Clearing all area of this font face */
        rectfill(buffer,xleftborder,face_area_top,xrightborder,buffer->h-status_line_height-1,back_color);

        textprintf_ex(buffer,font,xleft+1,face_area_top+top_margin-10,
                      colors[face_index],back_color,"%s",font_files[face_index]);

        /* Drawing grid */
        if (show_grid)
        {
            drawing_mode(DRAW_MODE_COPY_PATTERN,pattern_bitmap,0,0);
            for (i=0,x=xleft; i<=cells_in_row; i++,x+=(cell_size+1)) vline(buffer,x,ytop,ybottom,back_color);
            for (i=0,y=ytop; i<=cells_in_col; i++,y+=(cell_size+1)) hline(buffer,xleft,y,xright,back_color);
            drawing_mode(DRAW_MODE_SOLID,0,0,0);

            /* Drawing filled squares */
            drawing_mode(DRAW_MODE_COPY_PATTERN,pattern_bitmap,0,0);
            for (col=0,y=ytop,now_code=start_code; col<cells_in_col; col++,y+=(cell_size+1))
                for (row=0,x=xleft; row<cells_in_row; now_code++,row++,x+=(cell_size+1))
                    if (!gk_rend_has_glyph(rend,now_code))
                    {
                        rectfill(buffer,x+1,y+1,x+cell_size,y+cell_size,back_color);
                        if (gk_rend_has_character(rend,now_code))
                        {
                            drawing_mode(DRAW_MODE_SOLID,0,0,0);
                            rectfill(buffer,x+cell_size/4,y+cell_size/4,x+cell_size*3/4+1,y+cell_size*3/4+1,back_color);
                            drawing_mode(DRAW_MODE_COPY_PATTERN,pattern_bitmap,0,0);
                        }
                    }
            drawing_mode(DRAW_MODE_SOLID,0,0,0);
        }

        /* Rendering characters */
        if (!clipping_on)
            set_clip_rect(buffer,xleftborder,face_area_top,xrightborder,buffer->h-status_line_height-1);
        for (col=0,y=ytop,now_code=start_code; col<cells_in_col; col++,y+=(cell_size+1))
            for (row=0,x=xleft; row<cells_in_row; now_code++,row++,x+=(cell_size+1))
                if (gk_rend_has_glyph(rend,now_code))
                {
                    int cx = x+1+cell_size/2, cy = y+1+cell_size/2;
                    if (clipping_on) set_clip_rect(buffer,x+1,y+1,x+cell_size,y+cell_size);
                    gk_put_char_center(buffer,rend,now_code,cx,cy);
                    if (crosshair_on)
                    {
                        drawing_mode(DRAW_MODE_XOR,0,0,0);
                        hline(buffer,cx-cell_size/4,cy,cx+cell_size/4,crosshair_color);
                        vline(buffer,cx,cy-cell_size/4,cy+cell_size/4,crosshair_color);
                        drawing_mode(DRAW_MODE_SOLID,0,0,0);
                    }
                }
        set_clip_rect(buffer,0,0,buffer->w-1,buffer->h-1);
    }

    /*render_line_utf8(buffer,rends[0],"Hello, World!",screen_width/2,face_area_top);*/

    release_bitmap(buffer);
    if (buffer!=screen) blit(buffer,screen,0,0,0,0,buffer->w,buffer->h);
}


void exit_handler(void)
{
    if (allegro_installed)
    {
        if (logfile) fprintf(logfile,"Destroing bitmaps\n");
        if (pattern_bitmap) { destroy_bitmap(pattern_bitmap); pattern_bitmap = 0; }
        if (buffer && buffer!=screen) { destroy_bitmap(buffer); buffer = 0; }
        if (logfile) fprintf(logfile,"Shutting down Allegro\n");
        allegro_exit();
    }

    if (logfile)
    {
        fprintf(logfile,"shutdown clean\n");
        fclose(logfile);
        logfile = 0;
    }
}


int main(int argc,char **argv)
{
    int i,j;
    int done = 0;
    int changed = 1;
    int set_show_grid = 0, set_clipping = 0, set_crosshair = 0;
    int set_zoom_in = 0, set_zoom_out = 0;
    int openbrace_flag = 0, closebrace_flag = 0;
    int set_up = 0, set_down = 0, up_down_time = 0;
    int set_left = 0, set_right = 0;
    int set_page_up = 0, set_page_down = 0;
    int set_aa = 0;
    int set_ft_hinting = 0, set_tt_hinting = 0;
    int set_angle_cw = 0, set_angle_ccw = 0;
    int making_screenshot = 0, screenshot_index = 0;
    int set_italic_left = 0, set_italic_right = 0;
    int set_ratio_down = 0, set_ratio_up = 0;
    int set_bold_down = 0, set_bold_up = 0;
    int error = 0;

    atexit(exit_handler);
    logfile = fopen("compare_alleg.log","w");
    if (!logfile) exit(1);
    setbuf(logfile,0);
    fprintf(logfile,"compare_alleg - Glyph Keeper example program\n");
    face_num = argc-1;
    if (face_num > max_face_num) face_num = max_face_num;
    if (face_num > 0)
    {
        for (i=0; i<face_num; i++) font_files[i] = argv[i+1];
    }
    else
    {
        face_num = 1;
        font_files[0] = default_font_file;
    }
    /*if (face_num<1) { fprintf(logfile,"No fonts specified for comparison\n"); exit(1); }*/

    /* Installing Allegro, setting graphics mode. */
    if (install_allegro(SYSTEM_AUTODETECT,&errno,0)) { fprintf(logfile,"Error: can't init Allegro\n"); exit(1); }
    allegro_installed = 1;
    fprintf(logfile,"    Allegro initialized\n");

    read_config();

    /* Loading font faces, creating keeper and renderers. */
    gk_set_messenger(&messenger);
    gk_set_font_path(font_search_path);
    for (i=0; i<face_num; i++)
    {
        fprintf(logfile,"Loading font face from \"%s\"\n",font_files[i]);
        faces[i] = gk_load_face_from_file(font_files[i],0);
        if (!faces[i]) { fprintf(logfile,"Error: can't load font face from \"%s\"\n",font_files[i]); exit(1); }
        fprintf(logfile,"%s : \"%s\", %d glyphs / %d characters\n",font_files[i],gk_face_family(faces[i]),
            gk_face_number_of_glyphs(faces[i]),gk_face_number_of_characters(faces[i]));
    }
    keep = gk_create_keeper(0,glyph_cache_size);
    if (!keep) { fprintf(logfile,"Error: can't create glyph cache\n"); exit(1); }
    for (i=0; i<face_num; i++)
    {
        rends[i] = gk_create_renderer(faces[i],keep);
        if (!rends[i]) { fprintf(logfile,"Error: can't create glyph renderer for \"%s\"\n",font_files[i]); }
    }
    fprintf(logfile,"Glyph cache and renderers ready\n");

    fprintf(logfile,"Initializing Allegro\n");
    if (install_timer()) { fprintf(logfile,"Error: can't init Allegro timer\n"); exit(1); }
    fprintf(logfile,"    Allegro timer installed\n");
    if (install_keyboard())  { fprintf(logfile,"Error: can't init Allegro keyboard\n"); exit(1); }
    fprintf(logfile,"    Allegro keyboard installed\n");
    LOCK_VARIABLE(csecs);
    LOCK_FUNCTION(counter);
    if (install_int(counter,10)) { fprintf(logfile,"Error: can't install timer\n"); exit(1); }
    fprintf(logfile,"    Installed user timer (100 times/sec)\n");
    if (color_depth==0) color_depth = desktop_color_depth();
    if (color_depth==0) color_depth = 32;
    fprintf(logfile,"Setting graphics mode %dx%d, %d bit\n",screen_width,screen_height,color_depth);
    set_color_depth(color_depth);
    if (desktop_color_depth()>0) error = set_gfx_mode(GFX_AUTODETECT_WINDOWED,screen_width,screen_height,0,0);
    else error = set_gfx_mode(GFX_AUTODETECT,screen_width,screen_height,0,0);
    if (error)
        { fprintf(logfile,"Error: can't set graphics mode %d x %d x %d\n",screen_width,screen_height,color_depth); exit(1); }
    fprintf(logfile,"    Graphics mode %d x %d x %d set\n",screen_width,screen_height,color_depth);

    /* Preparing graphics stuff */
    max_cell_size = screen_width/face_num-5;
    back_color = makecol(0,0,0);
    grid_color = makecol(160,160,160);
    /*crosshair_color = makecol(200,0,0);*/
    crosshair_color = makecol(180,180,180);
    c1 = makecol(255,255,255);
    c2 = makecol(249,219,17);
    if (use_buffer)
    {
        buffer = create_bitmap(screen_width,screen_height);
        if (!buffer) { fprintf(logfile,"Error: can't create buffer bitmap\n"); exit(1); }
        fprintf(logfile,"Created %d x %d screen buffer\n",screen_width,screen_height);
    }
    else buffer = screen;
    pattern_bitmap = create_bitmap(4,4);
    if (!pattern_bitmap) { fprintf(logfile,"Error: can't create pattern bitmap\n"); exit(1); }
    clear_to_color(pattern_bitmap,back_color);
    for (i=0; i<4; i++) putpixel(pattern_bitmap,i,3-i,grid_color);
    fprintf(logfile,"Created pattern bitmap\n");
    for (j=0; j<num_colors; j++)
        colors[j] = makecol(colors_html[j]>>16,(colors_html[j]>>8)&0xFF,colors_html[j]&0xFF);
    fprintf(logfile,"Created color table\n");

    /* Setting initial settings for renderers */
    for (i=0; i<num_angles; i++) angles[i] = 3.14159265359*2*i/num_angles;
    fprintf(logfile,"Angle table set up\n");
    ratios[num_ratios] = 1.0;
    fprintf(logfile,"  ratios[%d] = %f\n",num_ratios,ratios[num_ratios]);
    for (i=0; i<num_ratios; i++)
    {
        ratios[num_ratios+i+1] = ratios[num_ratios+i] * 1.03;
        ratios[num_ratios-i-1] = ratios[num_ratios-i] / 1.03;
        fprintf(logfile,"  ratios[%d] = %f, ratios[%d] = %f\n",
            num_ratios+i+1,ratios[num_ratios+i+1],num_ratios-i-1,ratios[num_ratios-i-1]);
    }
    fprintf(logfile,"Ratio table set up\n");
    for (i=0; i<face_num; i++)
    {
        gk_rend_set_size_pixels(rends[i],text_size,text_size);
        if (!tt_hinting_on)
        {
            if (ft_hinting_on) gk_rend_set_hinting_force_autohint(rends[i]);
            else gk_rend_set_hinting_off(rends[i]);
        }
        gk_rend_set_text_color_rgb(rends[i],255,255,255);
        gk_rend_set_text_alpha_color(rends[i],0xFF000000|colors_html[i]);
        gk_rend_set_error_char(rends[i],'^');
    }
    fprintf(logfile,"Glyph renderers configured\n");

    while (!done)
    {
        if (changed) { update(); changed = 0; }
        rest(1);

        if (key[KEY_LEFT] && !set_right && !set_up && !set_down && start_code>0)
        {
            if (!set_left || (set_left && csecs-up_down_time >= ((key_shifts & KB_SHIFT_FLAG) ? 1 : 10) ))
            {
                set_left = 1;
                up_down_time = csecs;
                start_code--;
                if (start_code<0) start_code = 0;
                changed = 1;
            }
        }
        else set_left = 0;

        if (key[KEY_RIGHT] && !set_left && !set_up && !set_down && start_code<=(int)GK_MAX_UNICODE-cells_in_row*cells_in_col)
        {
            if (!set_right || (set_right && csecs-up_down_time >= ((key_shifts & KB_SHIFT_FLAG) ? 1 : 10) ))
            {
                set_right = 1;
                up_down_time = csecs;
                start_code++;
                if (start_code > (int)GK_MAX_UNICODE-cells_in_row*cells_in_col+1)
                    start_code = (int)GK_MAX_UNICODE-cells_in_row*cells_in_col+1;
                changed = 1;
            }
        }
        else set_right = 0;

        if (key[KEY_UP] && !set_down && start_code>0)
        {
            if (!set_up || (set_up && csecs-up_down_time >= ((key_shifts & KB_SHIFT_FLAG) ? 1 : 10) ))
            {
                set_up = 1;
                up_down_time = csecs;
                start_code -= cells_in_row;
                if (start_code<0) start_code = 0;
                changed = 1;
            }
        }
        else set_up = 0;

        if (key[KEY_DOWN] && !set_up && start_code<=(int)GK_MAX_UNICODE-cells_in_row*cells_in_col)
        {
            if (!set_down || (set_down && csecs-up_down_time >= ((key_shifts & KB_SHIFT_FLAG) ? 1 : 10) ))
            {
                set_down = 1;
                up_down_time = csecs;
                start_code += cells_in_row;
                if (start_code > (int)GK_MAX_UNICODE-cells_in_row*cells_in_col+1)
                    start_code = (int)GK_MAX_UNICODE-cells_in_row*cells_in_col+1;
                changed = 1;
            }
        }
        else set_down = 0;

        if (key[KEY_PGUP] && !set_page_down && !set_up && !set_down && start_code>0)
        {
            if (!set_page_up || (set_page_up && csecs-up_down_time>=10))
            {
                set_page_up = 1;
                up_down_time = csecs;
                start_code -= cells_in_row*(cells_in_col-1);
                if (start_code<0) start_code = 0;
                changed = 1;
            }
        }
        else set_page_up = 0;

        if (key[KEY_PGDN] && !set_page_up && !set_up && !set_down && start_code<=(int)GK_MAX_UNICODE-cells_in_row*cells_in_col)
        {
            if (!set_page_down || (set_page_down && csecs-up_down_time>= 10))
            {
                set_page_down = 1;
                up_down_time = csecs;
                start_code += cells_in_row*(cells_in_col-1);
                if (start_code > (int)GK_MAX_UNICODE-cells_in_row*cells_in_col+1)
                    start_code = (int)GK_MAX_UNICODE-cells_in_row*cells_in_col+1;
                changed = 1;
            }
        }
        else set_page_down = 0;

        if (key[KEY_COMMA] && !set_zoom_out && text_size>4)
        {
            double r;
            set_zoom_out = 1; text_size--; changed = 1;
            r = ratios[ratio_index+num_ratios];
            for (i=0; i<face_num; i++)
                gk_rend_set_size_subpixel(rends[i],((double)(text_size<<6))/r,((double)(text_size<<6))*r);
        }
        else set_zoom_out = 0;

        if (key[KEY_STOP] && !set_zoom_in && text_size<200)
        {
            double r;
            set_zoom_in = 1; text_size++; changed = 1;
            r = ratios[ratio_index+num_ratios];
            for (i=0; i<face_num; i++)
                gk_rend_set_size_subpixel(rends[i],((double)(text_size<<6))/r,((double)(text_size<<6))*r);
        }
        else set_zoom_in = 0;

        if (key[KEY_OPENBRACE]) 
        { 
            if (!openbrace_flag && cell_size>min_cell_size)
            {
                openbrace_flag = 1;
                cell_size -= (key_shifts & KB_SHIFT_FLAG) ? 10 : 1;
                if (cell_size < min_cell_size) cell_size = min_cell_size;
                changed = 1;
            }
        }
        else openbrace_flag = 0;

        if (key[KEY_CLOSEBRACE])
        {
            if (!closebrace_flag && cell_size<max_cell_size)
            {
                closebrace_flag = 1;
                cell_size += (key_shifts & KB_SHIFT_FLAG) ? 10 : 1;
                if (cell_size > max_cell_size) cell_size = max_cell_size;
                changed = 1;
            }
        }
        else closebrace_flag = 0;

        if (key[KEY_G])
            { if (!set_show_grid) { set_show_grid = 1; show_grid = 1 - show_grid; changed = 1; } }
        else set_show_grid = 0;

        if (key[KEY_C])
            { if (!set_clipping) { set_clipping = 1; clipping_on = 1 - clipping_on; changed = 1; } }
        else set_clipping = 0;

        if (key[KEY_R])
            { if (!set_crosshair) { set_crosshair = 1; crosshair_on = 1 - crosshair_on; changed = 1; } }
        else set_crosshair = 0;

        if (key[KEY_A])
        {
            if (!set_aa)
            {
                set_aa = 1;
                antialiasing_on = 1 - antialiasing_on;
                for (i=0; i<face_num; i++)
                    if (antialiasing_on) gk_rend_set_antialiasing_on(rends[i]);
                    else gk_rend_set_antialiasing_off(rends[i]);
                changed = 1;
            }
        }
        else set_aa = 0;

        if (key[KEY_F])
        {
            if (!set_ft_hinting)
            {
                set_ft_hinting = 1;
                tt_hinting_on = 0;
                ft_hinting_on = 1 - ft_hinting_on;
                for (i=0; i<face_num; i++)
                    if (ft_hinting_on) gk_rend_set_hinting_force_autohint(rends[i]);
                    else gk_rend_set_hinting_off(rends[i]);
                changed = 1;
            }
        }
        else set_ft_hinting = 0;

        if (key[KEY_T])
        {
            if (!set_tt_hinting)
            {
                set_tt_hinting = 1;
                ft_hinting_on = 0;
                tt_hinting_on = 1 - tt_hinting_on;
                for (i=0; i<face_num; i++)
                    if (tt_hinting_on) gk_rend_set_hinting_default(rends[i]);
                    else gk_rend_set_hinting_off(rends[i]);
                changed = 1;
            }
        }
        else set_tt_hinting = 0;

        if (key[KEY_1] && !set_angle_cw)
        {
            set_angle_ccw = 1;
            angle_index = (angle_index+1)%num_angles;
            for (i=0; i<face_num; i++)
                gk_rend_set_angle_in_radians(rends[i],angles[angle_index]);
            changed = 1;
        }
        else set_angle_ccw = 0;

        if (key[KEY_2] && !set_angle_ccw)
        {
            set_angle_cw = 1;
            angle_index = (angle_index+num_angles-1)%num_angles;
            for (i=0; i<face_num; i++)
                gk_rend_set_angle_in_radians(rends[i],angles[angle_index]);
            changed = 1;
        }
        else set_angle_cw = 0;

        if (key[KEY_3] && !set_italic_right && (italic > -45))
        {
            set_italic_left = 1;
            italic--;
            for (i=0; i<face_num; i++) gk_rend_set_italic_angle_in_degrees(rends[i],italic);
            changed = 1;
        }
        else set_italic_left = 0;

        if (key[KEY_4] && !set_italic_left && (italic < 45))
        {
            set_italic_right = 1;
            italic++;
            for (i=0; i<face_num; i++) gk_rend_set_italic_angle_in_degrees(rends[i],italic);
            changed = 1;
        }
        else set_italic_right = 0;


        if (key[KEY_5] && !set_ratio_up && (ratio_index > -num_ratios))
        {
            double r;
            set_ratio_down = 1;
            ratio_index--;
            r = ratios[ratio_index+num_ratios];
            for (i=0; i<face_num; i++)
                gk_rend_set_size_subpixel(rends[i],((double)(text_size<<6))/r,((double)(text_size<<6))*r);
            changed = 1;
        }
        else set_ratio_down = 0;

        if (key[KEY_6] && !set_ratio_down && (ratio_index < num_ratios))
        {
            double r;
            set_ratio_up = 1;
            ratio_index++;
            r = ratios[ratio_index+num_ratios];
            for (i=0; i<face_num; i++)
                gk_rend_set_size_subpixel(rends[i],((double)(text_size<<6))/r,((double)(text_size<<6))*r);
            changed = 1;
        }
        else set_ratio_up = 0;


        if (key[KEY_7] && !set_bold_up && (bold > -500))
        {
            set_bold_down = 1;
            bold -= 5;
            for (i=0; i<face_num; i++)
                gk_rend_set_bold_strength(rends[i],bold);
            changed = 1;
        }
        else set_bold_down = 0;

        if (key[KEY_8] && !set_bold_down && (bold < 500))
        {
            set_bold_up = 1;
            bold += 5;
            for (i=0; i<face_num; i++)
                gk_rend_set_bold_strength(rends[i],bold);
            changed = 1;
        }
        else set_bold_up = 0;

        if (key[KEY_HOME])
        {
            if (bold != 0 || ratio_index != 0 || italic != 0 || angle_index != 0)
            {
                if (bold != 0)
                {
                    bold = 0;
                    for (i=0; i<face_num; i++)
                        gk_rend_set_bold_strength(rends[i],bold);
                }

                if (ratio_index != 0)
                {
                    double r;
                    ratio_index = 0;
                    r = ratios[ratio_index+num_ratios];
                    for (i=0; i<face_num; i++)
                        gk_rend_set_size_subpixel(rends[i],((double)(text_size<<6))/r,((double)(text_size<<6))*r);
                }

                if (italic != 0)
                {
                    italic = 0;
                    for (i=0; i<face_num; i++) gk_rend_set_italic_angle_in_degrees(rends[i],italic);
                }

                if (angle_index != 0)
                {
                    angle_index = 0;
                    for (i=0; i<face_num; i++)
                        gk_rend_set_angle_in_radians(rends[i],angles[angle_index]);
                }

                changed = 1;
            }
        }


        if (key[KEY_ESC] || key[KEY_F10]) done = 1;

        if (key[KEY_F11])
        {
            if (!making_screenshot)
            {
                char filename[100] = "";
                BITMAP *bmp;
                PALETTE pal;
                making_screenshot = 1;
                sprintf(filename,"screen%04d.pcx",screenshot_index++);
                if (get_color_depth()==8) get_palette(pal);
                bmp = create_bitmap(screen_width,screen_height);
                if (bmp)
                {
                    blit(screen,bmp,0,0,0,0,screen_width,screen_height);
                    save_pcx(filename,bmp,(get_color_depth()==8)?pal:0);
                    destroy_bitmap(bmp);
                    bmp = 0;
                }
            }
        }
        else making_screenshot = 0;
    }

    return 0;
}
END_OF_MAIN()
