//LIC// FLXLab v2.5 - A program for running psychology experiments.  
//LIC// Copyright (C) 2010 Todd R. Haskell (todd.haskell@wwu.edu) 
//LIC// 
//LIC// Use and distribution is governed by the terms of the 
//LIC// GNU General Public License. Certain portions of the 
//LIC// program may be subject to other licenses as well. See 
//LIC// the file LICENSE.TXT for details.
//LIC// 
#define ALLEGRO_UNIX
#include "flxgraphics\screen.h"
#include <flxbase.h>
#include <linkflx.h>

FLX_LINKAGE FlxScreenObject *flx_screen;

long screen_width, screen_height;

/*****************************************************************************/

FLX_LINKAGE void FlxScreenObject::reset(void){
  string cur_function="FlxScreenObject::reset";
  int x, y;

  // set the default graphics mode
  d_graphics_mode=GFX_AUTODETECT_FULLSCREEN;
  /* To run FLXLab in a window, comment out the previous line and uncomment
     the following one */
  //d_graphics_mode=GFX_AUTODETECT_WINDOWED;

  // Set the default color depth. If this fails, it returns 0, and we 
  // don't try to set the color depth later when we initialize the screen.
  d_color_depth=desktop_color_depth();
  flx_data->write_message(FLX_DATADDEBUG,cur_function,"Setting screen color depth to "+flx_convert_to_string(d_color_depth));

  // set the default screen size
  if(get_desktop_resolution(&x,&y)!=0){
    // choose some sort of reasonable default
	x=640;
	y=480;
  }
  this->d_x_size->set_value(x);
  this->d_y_size->set_value(y);
  screen_width=x;
  screen_height=y;
  flx_data->write_message(FLX_DATADDEBUG,cur_function,"Setting screen size to "+flx_convert_to_string(x_size())+"x"+flx_convert_to_string(y_size()));

  d_is_initialized=false;

} /* FlxScreenObject::reset */

/*****************************************************************************/

FLX_LINKAGE bool FlxScreenObject::initialize(void){
  string cur_function="FlxScreenObject::initialize";
  bool do_set_mode;
  int x, y;

  /* check to see if any of the old settings are different from the new
     settings; we only set the graphics mode if there is at least one
     setting that has changed, or if the screen hasn't been initialized
     yet */
  if(!d_is_initialized){
    do_set_mode=true;
  } else if(d_color_depth && (d_color_depth!=desktop_color_depth())){
      do_set_mode=true;
  } else if(d_refresh_rate && (*d_refresh_rate!=get_refresh_rate())){
    do_set_mode=true;
  } else if( (get_desktop_resolution(&x,&y)==0) && ( (x!=x_size()) || (y!=y_size()) ) ){
    do_set_mode=true;
  } else {
    do_set_mode=false;
  }

  /* actually change the settings, if appropriate */
  d_is_initialized=true;
  if(!do_set_mode){
    flx_data->write_message(FLX_DATAINFO,cur_function,"Graphics mode is already appropriately set, skipping.");
    return true;
  } else {
    set_color_depth(d_color_depth);
    if(d_refresh_rate) request_refresh_rate(*d_refresh_rate);
    return display();
  }

} /* FlxScreenObject::initialize */

/*****************************************************************************/

FLX_LINKAGE bool FlxScreenObject::display(void){
  string cur_function="FlxScreenObject::display";

  if(!set_gfx_mode(d_graphics_mode,x_size(),y_size(),0,0)){
    flx_data->write_message(FLX_DATAINFO,cur_function,"Setting graphics mode to "+flx_convert_to_string(x_size())+" by "+flx_convert_to_string(y_size())+" at refresh rate of "+flx_convert_to_string(get_refresh_rate())+" Hz; driver is "+string(gfx_driver->name));
    return true;
  } else {
    flx_data->write_message(FLX_DATAERROR,cur_function,"Couldn't initialize graphics mode: "+string(allegro_error));
    return false;
  }

} /* FlxScreenObject::display */

/*****************************************************************************/

FLX_LINKAGE bool FlxScreenObject::minimize(void){
  string cur_function="FlxScreenObject::minimize";

  if(!set_gfx_mode(GFX_AUTODETECT_WINDOWED,200,200,0,0)){
    flx_data->write_message(FLX_DATADEBUG,cur_function,"Minimizing window");
    return true;
  } else {
    flx_data->write_message(FLX_DATAERROR,cur_function,"Couldn't minimize window: "+string(allegro_error));
    return false;
  }

} /* FlxScreenObject::minimize */

/*****************************************************************************/

bool clear_screen(void){
  string cur_function="clear_screen";

  flx_data->write_message(FLX_DATAHOOK,cur_function,"Clearing the screen");
  flx_screen->draw();
  return true;

} /* clear_screen */

/*****************************************************************************/

class ClearScreenEvent : public FlxEvent {
public:
  ClearScreenEvent(const string &name) : FlxEvent(name) {}
  ~ClearScreenEvent(){}
  void execute(void);
};

void ClearScreenEvent::execute(void){
  string cur_function="ClearScreenEvent::execute";

   flx_data->write_message(FLX_DATAEVENT,cur_function,d_name);
   this->do_generic_event_processing();
   flx_screen->draw(); 
}

/*****************************************************************************/

bool NewClearScreenEvent(string *name){
  string cur_function="NewClearScreenEvent";

  flx_data->write_message(FLX_DATASCRIPT,cur_function,"Creating new ClearScreenEvent '"+*name+"'");
  new ClearScreenEvent(*name);
  return true;

} /* NewClearScreenEvent */

/*****************************************************************************/

bool SetRefreshRate(long *rate){
  string cur_function="SetRefreshRate";

  flx_data->write_message(FLX_DATASCRIPT,cur_function,"Setting screen refresh rate");
  flx_screen->set_refresh_rate(rate);
  return true;

} /* SetRefreshRate */

/*****************************************************************************/

class RefreshCondition : public FlxCondition {
  bool d_triggered;
public:
  RefreshCondition(const string &name) : FlxCondition(name), d_triggered(false) {}
  ~RefreshCondition(void) {}
  bool evaluate(void);
  void reset(void) { d_triggered=false; }
};

bool RefreshCondition::evaluate(void){
  if(!d_triggered){
    vsync();
    d_triggered=true;
    return true;
  } else {
    return false;
  }
}

/*****************************************************************************/

FlxCondition *refresh_condition_parser(const string &keyword,int arg_count,string *arg_array){
  string cur_function="refresh_condition_parser";
  string name;

  if(arg_count!=0){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": Bad refresh condition");
      return NULL;
  } else {
    name=flx_unique_name();
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Creating new refresh condition named '"+name+"'");
    return new RefreshCondition(name);
  }

} /* refresh_condition_parser */

/*****************************************************************************/

bool set_graphics_mode(void){
string cur_function="set_graphics_mode";

 if(flx_screen->initialize()){
   flx_data->write_message(FLX_DATAHOOK,cur_function,"Setting graphics mode");
   return true;
 } else {
   flx_data->write_message(FLX_DATAERROR,cur_function,"Unable to set graphics mode");
   return false;
 }
 
} /* set_graphics_mode */

/*****************************************************************************/

void flx_screen_init(void){
  string cur_function="init_screen";
  FlxScalarObject<long> *flx_screen_width, *flx_screen_height;

  flx_add_command("ClearScreenEvent",NewClearScreenEvent);
  flx_add_command("RefreshRate",SetRefreshRate);
  flx_add_event_hook("set_graphics_mode",set_graphics_mode,FLX_EVENT_HOOK_PRESESSION);
  flx_add_event_hook("clear_screen",clear_screen,FLX_EVENT_HOOK_PRETRIAL);
  flx_add_condition_type("refresh",refresh_condition_parser);

  flx_data->write_message(FLX_DATAINFO,cur_function,"Creating screen");

  flx_screen=new FlxScreenObject("screen");
  
  flx_data->write_message(FLX_DATAINFO,cur_function,"Attempting to record screen width and height");
  flx_screen_width=new FlxScalarObject<long>("screen_width");
  flx_screen_width->set_value(&screen_width);
  flx_screen_height=new FlxScalarObject<long>("screen_height");
  flx_screen_height->set_value(&screen_height);
  
  /* we don't actually reset & initialize the screen until the module
  has been loaded, by means of a post-load callback function (see
  flxgraphics.cpp) */
  
} /* flx_screen_init */

/*****************************************************************************/
