//LIC// FLXLab v2.5 - A program for running psychology experiments.  
//LIC// Copyright (C) 2010 Todd R. Haskell (todd.haskell@wwu.edu) 
//LIC// 
//LIC// Use and distribution is governed by the terms of the 
//LIC// GNU General Public License. Certain portions of the 
//LIC// program may be subject to other licenses as well. See 
//LIC// the file LICENSE.TXT for details.
//LIC// 
//LIC//

#include <allegro.h>
#include <flxbase.h>
#include <agup.h>
#include "flxgui\dialog_lists.h"
#include "flxgui\bitmap_fonts.h"

FLX_LINKAGE bool *flx_close_on_click_true;
FLX_LINKAGE bool *flx_close_on_click_false;

/* We need this advance declaration for the macro below. */
char *list_callback(int,int *,int);

typedef char * (*ListCallbackFunc)(int,int *);

/* This macro defines a callback function list LIST_ID. It's only purpose is 
	to save some typing and simplify future changes to how the list callback
	works. */
#define DEFINE_CALLBACK(LIST_ID) \
char *list_callback ## LIST_ID(int index,int *list_size){ \
	return list_callback(index,list_size,LIST_ID); \
}

#define FLX_MAX_DIALOG_LISTS 3

/* Now create callback functions and pointers to list items for N lists,
	where N is the maximum number of lists we want to allow, summed across
	all dialog boxes in a script. */
DEFINE_CALLBACK(0)
DEFINE_CALLBACK(1)
DEFINE_CALLBACK(2)

/* Store pointers to the callback functions and item lists in arrays so
	that we can access them by using the list ID as the index. */
ListCallbackFunc list_callbacks[] = {list_callback0, list_callback1, list_callback2};
string *list_items[] = {NULL, NULL, NULL};
int num_list_items[] = {0, 0, 0};

/*****************************************************************************/

/* This is the actual callback function that tells Allegro what the items
	in the list are. */

char *list_callback(int index, int *list_size, int list_id){

	if(index>=0){
		return const_cast<char *>(list_items[list_id][index].c_str());
	} else {
		*list_size=num_list_items[list_id];
		return NULL;
	}

} /* list_callback */

/*****************************************************************************/

FLX_LINKAGE FlxDialogList::~FlxDialogList(void){

	if(list_items[d_list_id]){
	    delete[] list_items[d_list_id];
	    list_items[d_list_id]=NULL;
	}

} /* FlxDialogList::~FlxDialogList */

/*****************************************************************************/

FLX_LINKAGE void FlxDialogList::build(void *d,int &max_object_width,int &cur_y){
	string cur_function="FlxDialogList::build";
	DIALOG *dlg=static_cast<DIALOG *>(d);
	unsigned int num_items, widest_item, i;

	// store a pointer to the dialog element corresponding to this list
	d_element_ptr=d;

	/* Copy the list items to an array so they can be accessed without
		traversing the entire list. This complicates the code a little,
		but greatly increases the efficiency of the list callback function. */

	// count the number of items in the list
	num_items=0;
	while(d_item_list.item()){
		num_items++;
		d_item_list.advance();
	}
	d_item_list.rewind();
	num_list_items[d_list_id]=num_items;

	// allocate storage & do the actual copying
	if(list_items[d_list_id]){
		delete[] list_items[d_list_id];
		list_items[d_list_id]=NULL;
	}

	if(num_items>0) list_items[d_list_id]=new string[num_items];
	i=0;
	widest_item=0;
	while(d_item_list.item()){	
		list_items[d_list_id][i]= *(*(d_item_list.item()));
		if(list_items[d_list_id][i].size()>widest_item) widest_item=list_items[d_list_id][i].size();
		i++;
		d_item_list.advance();
	}
	d_item_list.rewind();

	// Now set up the dialog element itself
	dlg->proc=d_agup_list_proc;
	if(d_close_on_click) dlg->flags=D_EXIT;
	/* THIS CAST IS DANGEROUS - VOID * IS CONSIDERED A POINTER TO AN
	   OBJECT, AND THE STANDARD DOESN'T ALLOW CONVERSION BETWEEN
	   POINTERS TO OBJECTS AND POINTERS TO FUNCTIONS - BUT THERE'S
	   NOT MUCH I CAN DO ABOUT IT, SINCE THIS IS HOW ALLEGRO WORKS */
	dlg->dp=(void *) list_callbacks[d_list_id];
	//	dlg->dp=reinterpret_cast<void *>(list_callbacks[d_list_id]);
	dlg->x=0;
	dlg->y=cur_y;
	// set the size of the dialog object
	if(d_width==NULL){
	  dlg->w=static_cast<int>((widest_item+1.5)*flx_gui_font_width);
	} else {
		dlg->w=*d_width*flx_gui_font_width;
	}
	if(d_height==NULL){
		/* make the list tall enough for all the items or 10 items high,
			whichever is less */
	  dlg->h=static_cast<int>(((num_items>10 ? 10 : num_items) + .5)*flx_gui_font_height);
	} else {
	  dlg->h=static_cast<int>((*d_height+.5)*flx_gui_font_height);
	}

	// update parameters relating to the size of the overall dialog
	max_object_width=(dlg->w>max_object_width ? dlg->w : max_object_width);
	cur_y+=dlg->h;

} /* FlxDialogList::build */

/*****************************************************************************/

FLX_LINKAGE void FlxDialogList::update_value(int return_status){

	if(return_status==FLX_DIALOG_OK){
		if(d_item_list.item()){ // if there's at least one item in the list
			*d_var_text=list_items[d_list_id][static_cast<DIALOG *>(d_element_ptr)->d1];
		} else {
			*d_var_text="";
		}
	}

} /* FlxDialogList::update_value */

/*****************************************************************************/

bool NewDialogList(string *name,string *var_name){
  string cur_function="NewDialogList";
  static long num_lists=0;
  FlxScalarObject<string> *sop;
  FlxDialogList *dlp;

  if(num_lists<=FLX_MAX_DIALOG_LISTS){
    flx_data->write_message(FLX_DATASCRIPT,cur_function,"Creating new DialogList '"+*name+"'");
    sop=new FlxScalarObject<string>(*var_name);
    sop->set_value("");
    dlp=new FlxDialogList(*name);
    dlp->set_var_text(sop->value());
    dlp->set_list_id(num_lists);
    
    /* <num_lists> is used to create a unique identifier
       for each list. This identifier allows us to
       have a single callback function that can be do
       updates for any list object; it uses the
       identifier to know which list items to use. */
    num_lists++;
    return true;
  } else {
    flx_data->write_message(FLX_DATAERROR,cur_function,"Can't create new DialogList; only "+flx_convert_to_string(FLX_MAX_DIALOG_LISTS)+" allowed.");
    return false;
  }

} /* NewDialogList */

/*****************************************************************************/

bool DialogListItem(string *text){
	string cur_function="DialogListItem";
	FlxDialogList *dlp;

	if(flx_cur_dialog_object==NULL){
		flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": A DialogList object must be selected before you can create new list items.");
		return false;
	} else if((dlp=dynamic_cast<FlxDialogList *>(flx_cur_dialog_object))){
		flx_data->write_message(FLX_DATASCRIPT,cur_function,"Creating new list item");
		dlp->add_item(text);
		flx_add_scalar_source(flx_cur_dialog_object,text);
		return true;
	} else {
		flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": "+string(typeid(*flx_cur_dialog_object).name())+" can't have list items");
		return false;
	}

} /* DialogListItem */

/*****************************************************************************/

bool DialogListCloseOnClick(bool *flag){
	string cur_function="DialogListCloseOnClick";
	FlxDialogList *dlp;

	if(flx_cur_dialog_object==NULL){
		flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": A DialogList object must be selected before you can set the close-on-click attribute.");
		return false;
	} else if((dlp=dynamic_cast<FlxDialogList *>(flx_cur_dialog_object))){ 
		flx_data->write_message(FLX_DATASCRIPT,cur_function,"Setting close-on-click flag for '"+dlp->name()+"'");
		dlp->set_close_on_click(flag);
		return true;
	} else {
		flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": "+string(typeid(*flx_cur_dialog_object).name())+" isn't a dialog list object.");
		return false;
	}

} /* DialogListCloseOnClick */

/*****************************************************************************/

void flx_dialog_lists_init(void){
  FlxScalarObject<bool> *clicktrue, *clickfalse;

  flx_add_command("DialogList",NewDialogList);
  flx_add_command("DialogListItem",DialogListItem);
  flx_add_command("DialogListCloseOnClick",DialogListCloseOnClick);

  clicktrue=new FlxScalarObject<bool>("TRUE");
  clicktrue->set_value(true);
  flx_close_on_click_true=clicktrue->value();
  clickfalse=new FlxScalarObject<bool>("FALSE");
  clickfalse->set_value(false);
  flx_close_on_click_false=clickfalse->value();

} /* flx_dialog_lists_init */

/*****************************************************************************/
