//LIC// FLXLab v2.5 - A program for running psychology experiments.  
//LIC// Copyright (C) 2010 Todd R. Haskell (todd.haskell@wwu.edu) 
//LIC// 
//LIC// Use and distribution is governed by the terms of the 
//LIC// GNU General Public License. Certain portions of the 
//LIC// program may be subject to other licenses as well. See 
//LIC// the file LICENSE.TXT for details.
//LIC// 
#include <flxbase.h>
#include <iostream>
#include <fstream>
#include "flxsound/sound_event.h"
#include "flxsound/wav_files.h"
#include "flxsound/sound_driver.h"

using namespace std;

bool *default_wait_until_finished;

/*****************************************************************************/

class FlxPlaySoundEvent : public FlxSoundEvent {
  bool *d_wait_until_finished;
  long d_length;
public:
  FlxPlaySoundEvent(const string &name,string *file_name) : FlxSoundEvent(name,file_name), d_wait_until_finished(NULL) {
    flx_suspend_setting_updates();
    this->set_wait_until_finished(default_wait_until_finished);
    flx_resume_setting_updates();
  }
  ~FlxPlaySoundEvent(void){}
  bool update(void);
  void execute(void);
  void set_wait_until_finished(bool *wait_until_finished){
    flx_suspend_setting_updates();
    flx_change_scalar_setting(this,d_wait_until_finished,wait_until_finished,"FlxPlaySoundEvent::d_wait_until_finished"); 
    flx_resume_setting_updates();
  }
};

/*****************************************************************************/

bool FlxPlaySoundEvent::update(void){
  string cur_function="FlxPlaySoundEvent::update";
  ifstream *input_stream;
  Flx_WAV_File *wav_file;
  int bits, channels;
  long rate;
  long data_bytes;
  bool do_open_stream;
  bool valid_stream;
  long bytes_per_frame, frames;
  bool successful;

  input_stream=new ifstream(d_file_name->c_str(),ios::binary);
  if(!*input_stream){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Can't open WAV file");
    delete input_stream;
    return false;
  } 

  wav_file=new Flx_WAV_File();
  successful=wav_file->read(input_stream);  
  input_stream->close();
  delete input_stream;

  if(!successful){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Can't read WAV data from file");
    delete wav_file;
    return false;
  }

  wav_file->get_sound_params(bits,channels,rate,data_bytes);

  /* whether we need to open a new stream depends on a) whether we
     have already opened a stream, and if so b) whether the parameters of
     the sound we just loaded match those of the stream that is
     open */
  valid_stream=flx_sound_driver->is_valid_stream(d_stream);
  if(valid_stream && bits==d_buffer->bits() && channels==d_buffer->channels() && rate==d_buffer->rate()){
    do_open_stream=false;
  } else {
    do_open_stream=true;
  }

  // update the sound data
  d_buffer->set_params(bits,channels,rate,data_bytes,FLX_OUTPUT_STREAM);
  d_buffer->allocate_data();
  wav_file->get_sound_data(d_buffer->data());
  delete wav_file;

  if(do_open_stream){
    // if we already have an open stream, close it
    if(valid_stream){
      flx_sound_driver->close_stream(d_stream);
    }
    // open a new stream
    if(!(d_stream=flx_sound_driver->open_stream(d_buffer))){
      flx_data->write_message(FLX_DATAERROR,cur_function,"Can't open sound output stream");
      return false;
    }
  }

  bytes_per_frame=((bits-1)/8+1)*channels;
  frames=data_bytes/bytes_per_frame;
  d_length=frames*1000/rate;

  flx_data->write_message(FLX_DATADEBUG,cur_function,"Loaded WAV file and prepared sound output stream ("+flx_convert_to_string(bits)+" bits, "+flx_convert_to_string(channels)+" channels, "+flx_convert_to_string(rate)+" Hz, "+flx_convert_to_string(d_length)+" ms)");

  return true;

} /* FlxPlaySoundEvent::update */

/*****************************************************************************/

void FlxPlaySoundEvent::execute(void){
  string cur_function="FlxPlaySoundEvent::execute";

  if(flx_sound_driver->is_valid_stream(d_stream) && flx_sound_driver->is_active_stream(d_stream)){
    // stop the sound
    flx_sound_driver->stop_stream(d_stream);
  } else { 
    // start the sound
    if(!flx_sound_driver->start_stream(d_stream)){
      flx_data->write_message(FLX_DATAERROR,cur_function,"Can't start sound output stream");
    } else {
      // if appropriate, wait for the sound to finish playing
      if(*d_wait_until_finished){
	flx_delay(flx_using_microseconds ? d_length*1000 : d_length);
      }
    }
  }
  
  flx_data->write_message(FLX_DATAEVENT,cur_function,d_name);
  this->do_generic_event_processing();

} /* FlxPlaySoundEvent::execute */

/*****************************************************************************/

bool NewPlaySoundEvent(string *name,string *file_name){
  string cur_function="NewPlaySoundEvent";
  FlxPlaySoundEvent *pse;

  flx_data->write_message(FLX_DATASCRIPT,cur_function,"Creating new PlaySoundEvent '"+*name+"'");
  pse=new FlxPlaySoundEvent(*name,file_name);
  return true;

} /* NewPlaySoundEvent */

/*****************************************************************************/

bool SetWaitUntilFinished(bool *flag){
  string cur_function="SetWaitUntilFinished";
  FlxPlaySoundEvent *pse;

  if(flx_cur_event==NULL){
    flx_data->write_message(FLX_DATASCRIPT,cur_function,"Setting default wait-until-finished attribute");
    flx_change_global_setting(&default_wait_until_finished,flag,"default_wait_until_finished");
    return true;
  } else if((pse=dynamic_cast<FlxPlaySoundEvent *>(flx_cur_event))){
    flx_data->write_message(FLX_DATASCRIPT,cur_function,"Setting wait-until-finished attribute of '"+pse->name()+"'");
    pse->set_wait_until_finished(flag);
    return true;
  } else {
    flx_data->write_message(FLX_DATAERROR,cur_function,"Error at script line "+flx_convert_to_string(flx_cur_script_line)+": "+string(typeid(*flx_cur_event).name())+" doesn't have a wait-until-finished attribute.");
    return false;
  }

} /* SetWaitUntilFinished */

/*****************************************************************************/

void flx_play_sound_init(void){
  string cur_function="flx_play_sound_init";

  flx_add_command("PlaySoundEvent",NewPlaySoundEvent);
  flx_add_command("WaitUntilFinished",SetWaitUntilFinished);

  default_wait_until_finished=flx_bool(true);
  
} /* flx_play_sound_init */

/*****************************************************************************/



