//LIC// FLXLab v2.5 - A program for running psychology experiments.  
//LIC// Copyright (C) 2010 Todd R. Haskell (todd.haskell@wwu.edu) 
//LIC// 
//LIC// Use and distribution is governed by the terms of the 
//LIC// GNU General Public License. Certain portions of the 
//LIC// program may be subject to other licenses as well. See 
//LIC// the file LICENSE.TXT for details.
//LIC// 
//LIC//
#define GLYPH_TARGET GLYPH_TARGET_ALLEGRO 
#include <glyph.h>
#include <flxbase.h>
#include <linkflx.h>

/* This stores the names and file locations of the fonts that are available 
   to a script. */
FlxTriplexList<string,string,string> font_name_list;
/* This stores the name and data of fonts that have been used at least once
   in a script. Note that this will always be a subset of the fonts in the
   font name list. */
FlxTriplexList<GLYPH_FACE *,string,string> font_object_list;
GLYPH_REND *font_renderer;
GLYPH_KEEP *font_cache;
FONT *allegro_font;

string font_list_file;
bool append_to_font_list;

bool set_font_properties(string font,long size,bool antialiasing){
  string cur_function="set_font_properties";
  string *font_file;
  GLYPH_FACE *font_ptr, **font_handle;

  flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Setting font properties to "+flx_convert_to_string(size)+" point "+font+" "+(antialiasing ? "antialiased": "non-antialiased"));

  /*** GET A POINTER TO A FONT FACE OBJECT ***/
  /* If the font has already been loaded, just get a pointer to it. */
  if((font_handle=font_object_list.find_item2(font))){
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Font '"+font+"' is already loaded, getting pointer.");
    font_ptr=*font_handle;
  } else {
    /* Font is not already loaded, check to make sure the name is registered. */
    if(!(font_file=font_name_list.find_item2(font))){
       flx_data->write_message(FLX_DATAERROR,cur_function,"Can't find font '"+font+"'");
       return false;
    } else {
      /* Name is registered, try to load the font. */
      if(!(font_ptr=gk_load_face_from_file(font_file->c_str(),0))){
	flx_data->write_message(FLX_DATAERROR,cur_function,"Couldn't load font '"+font+"'");
	return false;
      } else {
	flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Loaded font '"+font+"'.");
	font_object_list.insert(font_ptr,font,flx_cur_scope_tag); 
      }
    }
  }

  /*** SET THE PROPERTIES OF THE RENDERER ***/
  gk_rend_set_face(font_renderer,font_ptr);
  // make sure the provided size is in a reasonable range
  if(size<1 || size>500){
    flx_data->write_message(FLX_DATAERROR,cur_function,"Font sizes less than 1 or greater than 500 are not allowed");
    return false;
  }
  if(!gk_rend_set_size_pixels(font_renderer,size,size)){
    flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Can't set size of font '"+font+"' to "+flx_convert_to_string(size)+".");
    return false;
  }
  if(antialiasing){
    gk_rend_set_antialiasing_on(font_renderer);
  } else {
    gk_rend_set_antialiasing_off(font_renderer);
  }

  flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Successfully set properties of font");

  return true;

} /* set_font_properties */

/*****************************************************************************/

FONT *get_allegro_font(void){
  string cur_function="get_allegro_font";

  flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Getting a pointer to an allegro font");
  return allegro_font;
  
} /* get_allegro_font */

/*****************************************************************************/

long get_text_height(void){
  /* this only returns the height above the baseline, not overall
  height, so that alignment works correctly */
  string cur_function="get_text_height";

  flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Computing text height");
  return gk_rend_ascender_pixels(font_renderer);

} /* get_text_height */

/*****************************************************************************/
		    
long get_text_width(string text){
  string cur_function="get_text_width";

  flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Computing text width of '"+text+"'");
  return gk_text_width_utf8(font_renderer,text.c_str());

} /* get_text_width */

/*****************************************************************************/

void flx_add_font_directory(const string &directory,int data_level){
  string cur_function="flx_add_font_directory";
  string formatted_directory, font_name, *font_file;
  FlxLinkedList<string> font_files;
  GLYPH_FACE *font_ptr;
  char *name_string;
  unsigned int i;

  /* if the directory name doesn't already end with a path separator, add it */
  if(directory.size() && directory.substr(directory.size()-1,1)!=flx_path_separator){
    formatted_directory=directory+flx_path_separator;
  } else {
    formatted_directory=directory;
  }

  flx_data->write_message(data_level,cur_function,"Adding font directory '"+formatted_directory+"'");
  flx_get_directory_entries(formatted_directory,font_files);
  flx_data->new_queue(font_list_file);
  flx_data->queue()->set_append_to_existing_file(append_to_font_list);
  append_to_font_list=true;
  while((font_file=font_files.item())){
    flx_data->write_message(FLX_DATADEBUG,cur_function,"Checking file "+*font_file);
    if((font_ptr=gk_load_face_from_file((formatted_directory+*font_file).c_str(),0))){
      if((name_string=const_cast<char *>(gk_face_postscript_name(font_ptr))) && ((font_name=name_string)!="UNAVAILABLE")){
	/* Remove any hyphens from the name. This is because we want to
	   be able to just remove the spaces from any name the user
	   provides, and arrive at the name we have registered. */
	for(i=0;i<font_name.size();i++){
	  if(font_name[i]=='-'){
	    font_name.erase(i,1);
	    i--;
	  }
	}
	if(!font_name_list.find_item2(font_name)){
	  flx_data->write_message(FLX_DATADATA,cur_function,font_name+"\t\""+formatted_directory+*font_file+"\"");
	  font_name_list.insert(formatted_directory+*font_file,font_name,flx_cur_scope_tag);
	} else {
	  flx_data->write_message(FLX_DATADDEBUG,cur_function,"Font '"+font_name+"' already registered, skipping.");
	}
      } else {
	flx_data->write_message(FLX_DATADDEBUG,cur_function,"Can't get font name from file '"+*font_file+"', does not appear to be a Postscript or TrueType font, skipping.");
      }
      gk_unload_face(font_ptr);
    } else {
      flx_data->write_message(FLX_DATADDEBUG,cur_function,"File '"+*font_file+"' cannot be loaded as a font, skipping.");
    }
    font_files.advance();
  }
  flx_data->delete_queue();

} /* flx_add_font_directory */

/*****************************************************************************/

bool AddFontDirectory(string *path){
 
  flx_add_font_directory(*path,FLX_DATASCRIPT);
  return true;
 
} /* AddFontDirectory */

/*****************************************************************************/
 
void remove_font_names_by_scope(const string &tag){
  string cur_function="remove_font_names_by_scope";
  string *font_name;

 flx_data->write_message(FLX_DATADDEBUG,cur_function,"Removing font names with scope '"+tag+"'");
 while(font_name_list.advance_to_item3(tag)){
   font_name=font_name_list.item2();
   flx_data->write_message(FLX_DATADDDEBUG,cur_function,"Removing font '"+*font_name+"'");
	font_name_list.remove();
 }
 font_name_list.rewind();

} /* remove_font_names_by_scope */
                                    
/*****************************************************************************/
 
void remove_font_objects_by_scope(const string &tag){
  string cur_function="remove_font_objects_by_scope";
  GLYPH_FACE *gf;

 flx_data->write_message(FLX_DATADDEBUG,cur_function,"Removing font objects with scope '"+tag+"'");
 while(font_object_list.advance_to_item3(tag)){
   gf=*(font_object_list.item());
   gk_unload_face(gf);
   font_object_list.remove();
 }
 font_object_list.rewind();

} /* remove_font_objects_by_scope */

/*****************************************************************************/
                                    
void font_init_callback(void){

  gk_library_init();
  /* The font cache is a feature of glyph keeper which stores the image
     of recently-rendered characters so they don't have to be rendered
     each time they are used */
  font_cache=gk_create_keeper(0,0);
  font_renderer=gk_create_renderer(0,font_cache);
  allegro_font=gk_create_allegro_font(font_renderer);

  font_list_file=flxlab_dir+"resources"+flx_path_separator+"font_list.txt";
  append_to_font_list=false;

  // register the default font directory
  flx_add_font_directory(flxlab_dir+"fonts",FLX_DATADEBUG);

} /* font_init_callback */

/*****************************************************************************/

void fonts_exit(void){

	gk_library_cleanup();

} /* fonts_exit */

/*****************************************************************************/

void flx_fonts_init(void){
  string cur_function="fonts_init";
  
  flx_add_command("AddFontDirectory",AddFontDirectory);
  flx_add_scope_cleanup_hook("remove_font_objects_by_scope",remove_font_objects_by_scope);
  flx_add_scope_cleanup_hook("remove_font_names_by_scope",remove_font_names_by_scope);
  flx_add_scope_exit_hook("fonts_exit",fonts_exit);

} /* flx_fonts_init */

/*****************************************************************************/

